/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include <string.h>
#include "main.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

ald_uart_handle_t h_uart;

uint8_t tx_buf[8];
uint8_t rx_buf[8];
uint8_t sflag = 0;
uint8_t rflag = 0;
static void uart_init(void);

/**
  * @brief  Initializate pin of uart module.
  * @retval None
  */
void uart_pin_init(void)
{
    ald_gpio_init_t x;

    /* Initialize tx pin */
    x.mode  = ALD_GPIO_MODE_OUTPUT;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.func  = ALD_GPIO_FUNC_3;
    ald_gpio_init(GPIOB, GPIO_PIN_10, &x);

    /* Initialize rx pin */
    x.mode  = ALD_GPIO_MODE_INPUT;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.func  = ALD_GPIO_FUNC_3;
    ald_gpio_init(GPIOB, GPIO_PIN_11, &x);

    return;
}

/**
  * @brief  Send message complete.
  * @param  arg: Pointer to ald_uart_handle_t structure.
  * @retval None.
  */
void uart_send_complete(ald_uart_handle_t *arg)
{
    sflag = 0x1;;
}

/**
  * @brief  Receive a message complete.
  * @param  arg: Pointer to ald_uart_handle_t structure.
  * @retval None.
  */
void uart_recv_complete(ald_uart_handle_t *arg)
{
    rflag = 0x1;
}

/**
  * @brief  Occurs error.
  * @param  arg: Pointer to ald_uart_handle_t structure.
  * @retval None.
  */
void uart_error(ald_uart_handle_t *arg)
{
    ald_rmu_reset_periperal(ALD_RMU_PERH_UART0);
    uart_init();
}

/**
  * @brief  UART inint.
  * @param  None.
  * @retval None.
  */
void uart_init(void)
{
    /* Initialize uart */
    h_uart.perh             = UART0;
    h_uart.init.baud        = 115200;
    h_uart.init.word_length = ALD_UART_WORD_LENGTH_8B;
    h_uart.init.stop_bits   = ALD_UART_STOP_BITS_1;
    h_uart.init.parity      = ALD_UART_PARITY_NONE;
    h_uart.init.mode        = ALD_UART_MODE_UART;
    h_uart.init.fctl        = ALD_UART_HW_FLOW_CTL_DISABLE;
    h_uart.tx_cplt_cbk      = uart_send_complete;
    h_uart.rx_cplt_cbk      = uart_recv_complete;
    h_uart.error_cbk        = uart_error;
    ald_uart_init(&h_uart);
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    /* Initialize ALD */
    ald_cmu_init();
    ald_cmu_pll_config(ALD_CMU_PLL_INPUT_HRC, ALD_CMU_PLL_OUTPUT_48M);
    ald_cmu_clock_config(ALD_CMU_CLOCK_PLL, 48000000);
    ald_cmu_perh_clock_config(ALD_CMU_PERH_ALL, ENABLE);
    ald_mcu_irq_config(UART0_IRQn, 1, 1, ENABLE);

    /* clear ald_uart_handle_t structure */
    memset(&h_uart, 0x0, sizeof(h_uart));
    /* Initialize pin */
    uart_pin_init();
    uart_init();

    /* Initialize tx_buf */
    memset(tx_buf, 0x55, sizeof(tx_buf));

    /* Send a message */
    ald_uart_send_by_dma(&h_uart, tx_buf, 8, 0);
    while(!sflag);
    sflag = 0x0;

    /* Receive a message */
    ald_uart_recv_by_dma(&h_uart, rx_buf, 8, 0);

    while (1)
    {
        if (0x1 == rflag)
        {
            rflag = 0;
            ald_uart_send_by_dma(&h_uart, rx_buf, 8, 0);
            while(!sflag);
            sflag = 0x0;
            memset((void*)rx_buf, 0x00, sizeof(rx_buf));
            ald_uart_recv_by_dma(&h_uart, rx_buf, 8, 0);
        }
    }
}

/**
  * @}
  */
/**
  * @}
  */
