/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include "main.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

ald_timer_handle_t h_tim;
ald_timer_clock_config_t tim_clock;
ald_timer_oc_init_t tim_ocinit;

/**
  * @brief  Initializate pin of tim module.
  * @retval None
  */
void tim_pin_init(void)
{
    ald_gpio_init_t x;

    /* Initialize tim0 ch1 pin */
    x.mode  = ALD_GPIO_MODE_OUTPUT;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.func  = ALD_GPIO_FUNC_2;
    ald_gpio_init(GPIOA, GPIO_PIN_8, &x);

    x.func = ALD_GPIO_FUNC_1;
    ald_gpio_init(GPIOC, GPIO_PIN_12, &x);
    ald_gpio_write_pin(GPIOC, GPIO_PIN_12, 1);
}

/**
  * @brief  This is main function
  * @retval Status
  */
int main()
{
    /* Initialize ALD */
    ald_cmu_init();
    /* Configure system clock */
    ald_cmu_pll_config(ALD_CMU_PLL_INPUT_HRC, ALD_CMU_PLL_OUTPUT_48M);
    ald_cmu_clock_config(ALD_CMU_CLOCK_PLL, 48000000);
    ald_cmu_perh_clock_config(ALD_CMU_PERH_ALL, ENABLE);

    /* Initialize AD16C4T0 */
    h_tim.perh           = AD16C4T0;
    h_tim.init.prescaler = 4699;
    h_tim.init.mode      = ALD_TIMER_CNT_MODE_CENTER1;
    h_tim.init.period    = 9999;
    h_tim.init.clk_div   = ALD_TIMER_CLOCK_DIV1;
    h_tim.init.re_cnt    = 0;
    ald_timer_pwm_init(&h_tim);

    /* Initialize pin */
    tim_pin_init();

    /* Initialize clock source */
    tim_clock.source = ALD_TIMER_SRC_INTER;
    ald_timer_config_clock_source(&h_tim, &tim_clock);

    /* Common configuration for all channels */
    tim_ocinit.oc_mode      = ALD_TIMER_OC_MODE_PWM1;
    tim_ocinit.oc_polarity  = ALD_TIMER_OC_POLARITY_HIGH;
    tim_ocinit.oc_fast_en   = DISABLE;
    tim_ocinit.ocn_polarity = ALD_TIMER_OCN_POLARITY_HIGH;
    tim_ocinit.ocn_idle     = ALD_TIMER_OCN_IDLE_RESET;
    tim_ocinit.oc_idle      = ALD_TIMER_OC_IDLE_RESET;

    /* Set the pulse value for channel 1 */
    tim_ocinit.pulse = 50;
    ald_timer_oc_config_channel(&h_tim, &tim_ocinit, ALD_TIMER_CHANNEL_1);

    ald_mcu_irq_config(AD16C4T0_CC_IRQn, 1, 1, ENABLE);
    ald_timer_interrupt_config(&h_tim, ALD_TIMER_IT_CC1, ENABLE);

    /* Start input pwm from tim0 channel 1 */
    ald_timer_pwm_start(&h_tim, ALD_TIMER_CHANNEL_1);

    while (1);
}

void AD16C4T0_CC_Handler(void)
{
    uint32_t reg_cr = h_tim.perh->CON1;
    uint32_t reg = h_tim.perh->IFM;

    /* Capture or compare 1 event */
    if (reg & (0X1 << 1))
    {
        ald_timer_clear_flag_status(&h_tim, ALD_TIMER_FLAG_CC1);

        if (reg_cr & 0X10)
        {
            ald_gpio_toggle_pin(GPIOC, GPIO_PIN_12);
        }
    }
}

/**
  * @}
  */
/**
  * @}
  */
