/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include "main.h"
#include <string.h>

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

ald_pis_handle_t h_pis;
ald_timer_handle_t h_tim1;
ald_timer_handle_t h_tim2;
ald_timer_clock_config_t tim_clock;

/**
  * @brief  Initializate I/O.
  * @retval None
  */
void led_pin_init(void)
{
    ald_gpio_init_t x;

    /* Initialize PA4 as output */
    x.mode  = ALD_GPIO_MODE_OUTPUT;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.func  = ALD_GPIO_FUNC_1;
    ald_gpio_init(GPIOC, GPIO_PIN_12, &x);
}

/**
  * @brief  Initializate timer module.
  * @retval None
  */
void pis_tim_init(void)
{
    /* clear timer_handle_t structure */
    memset(&h_tim1, 0, sizeof(ald_timer_handle_t));
    /* clear timer_handle_t structure */
    memset(&h_tim2, 0, sizeof(ald_timer_handle_t));
    /* clear timer_clock_config_t structure */
    memset(&tim_clock, 0, sizeof(ald_timer_clock_config_t));

    /* Initialize AD16C4T0 */
    h_tim1.perh           = AD16C4T0;
    h_tim1.init.prescaler = 469;
    h_tim1.init.mode      = ALD_TIMER_CNT_MODE_UP;
    h_tim1.init.period    = 99999;
    h_tim1.init.clk_div   = ALD_TIMER_CLOCK_DIV1;
    h_tim1.init.re_cnt    = 0;
    ald_timer_base_init(&h_tim1);

    /* Initialize GP32C4T0 */
    h_tim2.perh           = GP32C4T0;
    h_tim2.init.prescaler = 0x01;
    h_tim2.init.mode      = ALD_TIMER_CNT_MODE_UP;
    h_tim2.init.period    = 0x300;
    h_tim2.init.clk_div   = ALD_TIMER_CLOCK_DIV1;
    h_tim2.init.re_cnt    = 0;
    ald_timer_base_init(&h_tim2);

    /* Initialize clock source */
    tim_clock.source   = ALD_TIMER_SRC_ITR0;
    tim_clock.psc      = ALD_TIMER_ETR_PSC_DIV1;
    tim_clock.filter   = 0;
    tim_clock.polarity = ALD_TIMER_CLK_POLARITY_RISE;
    ald_timer_config_clock_source(&h_tim2, &tim_clock);
}

/**
  * @brief  Start timer0 and timer2 module.
  * @retval None
  */
void pis_tim_start(void)
{
    ald_timer_base_start(&h_tim1);
    ald_timer_base_start(&h_tim2);
}

/**
  * @brief  Stop timer1 module.
  * @retval None
  */
void pis_tim_stop(void)
{
    ald_timer_base_stop(&h_tim1);
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    /* Initialize ALD */
    ald_cmu_init();
    ald_cmu_pll_config(ALD_CMU_PLL_INPUT_HRC, ALD_CMU_PLL_OUTPUT_48M);
    ald_cmu_clock_config(ALD_CMU_CLOCK_PLL, 48000000);
    ald_cmu_perh_clock_config(ALD_CMU_PERH_ALL, ENABLE);
    
    led_pin_init();

    /* Clear pis_handle_t struct */
    memset(&h_pis, 0, sizeof(ald_pis_handle_t));

    /* Initialize timer */
    pis_tim_init();

    /* Initialize PIS */
    h_pis.perh                 = PIS;
    h_pis.init.producer_src    = ALD_PIS_TIMER0_UPDATA;
    h_pis.init.producer_clk    = ALD_PIS_CLK_PCLK1;
    h_pis.init.producer_edge   = ALD_PIS_EDGE_NONE;
    h_pis.init.producer_signal = ALD_PIS_OUT_PULSE;
    h_pis.init.consumer_trig   = ALD_PIS_CH0_TIMER2_ITR0;
    h_pis.init.consumer_clk    = ALD_PIS_CLK_PCLK1;
    ald_pis_create(&h_pis);

    pis_tim_start();

    while (1)
    {
        ald_delay_ms(1000);
        ald_gpio_toggle_pin(GPIOC, GPIO_PIN_12);
    }
}
/**
  * @}
  */

/**
  * @}
  */
