/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include "main.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
ald_timer_handle_t h_tim;
ald_dma_handle_t hperh;
ald_timer_clock_config_t clock;
ald_timer_ic_init_t init;
ald_timer_slave_config_t slave_config;

/* duty/cycle data buffer */
#define BUFFER_SIZE 32
uint16_t duty_buffer[BUFFER_SIZE] __attribute__((aligned(4))) = {0};
uint16_t cycle_buffer[BUFFER_SIZE] __attribute__((aligned(4))) = {0};

/**
  * @brief  Initializate pin of tim module.
  * @retval None
  */
void timer_pin_init(void)
{
    ald_gpio_init_t x;

    /* Initialize tim0 ch1 pin */
    x.mode  = ALD_GPIO_MODE_INPUT;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.func  = ALD_GPIO_FUNC_2;
    ald_gpio_init(TIMER_INPUTCAPTURE_PORT, TIMER_INPUTCAPTURE_PIN, &x);
}

/**
  * @brief  Initializate DMA chanel0 and channel1.
  * @retval None
  */
void init_dma(void)
{
    hperh.perh = DMA0;
    ald_dma_config_struct(&hperh);

    hperh.config.src            = (void *)(&AD16C4T0->CCVAL1);
    hperh.config.src_data_width = ALD_DMA_DATA_SIZE_HALFWORD;
    hperh.config.src_inc        = ALD_DMA_DATA_INC_DISABLE;
    hperh.config.dst            = (void *)cycle_buffer;
    hperh.config.dst_data_width = ALD_DMA_DATA_SIZE_HALFWORD;
    hperh.config.dst_inc        = ALD_DMA_DATA_INC_ENABLE;
    hperh.config.size           = BUFFER_SIZE;
    hperh.config.priority       = ALD_DMA_LOW_PRIORITY;
    hperh.config.R_power        = ALD_DMA_R_POWER_32;
    hperh.config.dir            = ALD_DMA_DIR_TO_SRAM;
    hperh.config.circle_mode    = ENABLE;
    hperh.config.msel           = ALD_DMA_MSEL_TIMER0;
    hperh.config.msigsel        = ALD_DMA_MSIGSEL_TIMER_CH1;

    hperh.cplt_tc_cbk           = NULL;
    hperh.cplt_tc_arg           = NULL;
    hperh.config.channel        = 0;
    ald_dma_config(&hperh);

    hperh.perh = DMA0;
    ald_dma_config_struct(&hperh);

    hperh.config.src                = (void *)(&AD16C4T0->CCVAL2);
    hperh.config.src_data_width     = ALD_DMA_DATA_SIZE_HALFWORD;
    hperh.config.src_inc            = ALD_DMA_DATA_INC_DISABLE;
    hperh.config.dst                = (void *)duty_buffer;
    hperh.config.dst_data_width     = ALD_DMA_DATA_SIZE_HALFWORD;
    hperh.config.dst_inc            = ALD_DMA_DATA_INC_ENABLE;
    hperh.config.size               = BUFFER_SIZE;
    hperh.config.priority           = ALD_DMA_LOW_PRIORITY;
    hperh.config.R_power            = ALD_DMA_R_POWER_32;
    hperh.config.dir                = ALD_DMA_DIR_TO_SRAM;
    hperh.config.circle_mode        = ENABLE;

    hperh.config.msel               = ALD_DMA_MSEL_TIMER0;
    hperh.config.msigsel            = ALD_DMA_MSIGSEL_TIMER_CH2;

    hperh.cplt_tc_cbk               = NULL;
    hperh.cplt_tc_arg               = NULL;
    hperh.config.channel            = 1;
    ald_dma_config(&hperh);
}

/**
  * @brief  Main function
  * @retval Status
  */
int main()
{
    /* Initialize ALD */
    ald_cmu_init();
    ald_cmu_pll_config(ALD_CMU_PLL_INPUT_HRC, ALD_CMU_PLL_OUTPUT_48M);
    ald_cmu_clock_config(ALD_CMU_CLOCK_PLL, 48000000);
    ald_cmu_perh_clock_config(ALD_CMU_PERH_ALL, ENABLE);
    /* Initialize pin and set the default state*/
    timer_pin_init();

    /* Initialize AD16C4T0 */
    h_tim.perh           = AD16C4T0;
    h_tim.init.prescaler = 0;
    h_tim.init.mode      = ALD_TIMER_CNT_MODE_UP;
    h_tim.init.period    = 0xFFFF;
    h_tim.init.clk_div   = ALD_TIMER_CLOCK_DIV1;
    h_tim.init.re_cnt    = 0;
    h_tim.capture_cbk    = NULL;
    ald_timer_ic_init(&h_tim);

    /* Initialize clock source */
    clock.source     = ALD_TIMER_SRC_INTER;
    ald_timer_config_clock_source(&h_tim, &clock);

    /* Initialize input capture channel */
    init.polarity = ALD_TIMER_IC_POLARITY_RISE;
    init.sel      = ALD_TIMER_IC_SEL_DIRECT;
    init.psc      = ALD_TIMER_IC_PSC_DIV1;
    init.filter   = 3;
    ald_timer_ic_config_channel(&h_tim, &init, ALD_TIMER_CHANNEL_1);

    /* Initialize input capture channel */
    init.polarity = ALD_TIMER_IC_POLARITY_FALL;
    init.sel      = ALD_TIMER_IC_SEL_INDIRECT;
    init.psc      = ALD_TIMER_IC_PSC_DIV1;
    init.filter   = 2;
    ald_timer_ic_config_channel(&h_tim, &init, ALD_TIMER_CHANNEL_2);

    slave_config.mode       = ALD_TIMER_MODE_RESET;
    slave_config.input      = ALD_TIMER_TS_TI1FP1;
    slave_config.polarity   = ALD_TIMER_CLK_POLARITY_RISE;
    slave_config.psc        = ALD_TIMER_ETR_PSC_DIV1;
    slave_config.filter     = 4;
    ald_timer_slave_config_sync(&h_tim, &slave_config);

    init_dma();

    TIMER_CCx_DMA_REQ_CCx(&h_tim);
    ald_timer_dma_req_config(&h_tim, ALD_TIMER_DMA_CC1, ENABLE);
    ald_timer_dma_req_config(&h_tim, ALD_TIMER_DMA_CC2, ENABLE);

    /* Start input capture */
    ald_timer_ic_start(&h_tim, ALD_TIMER_CHANNEL_1);
    ald_timer_ic_start(&h_tim, ALD_TIMER_CHANNEL_2);

    while (1)
    {
        ald_delay_ms(1000);
    }
}
/**
  * @}
  */
/**
  * @}
  */
