/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include "main.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
ald_timer_handle_t h_tim;
ald_timer_clock_config_t clock;
ald_timer_ic_init_t init;

/* Captured Values */
uint32_t value1 = 0;
uint32_t value2 = 0;
uint32_t diff = 0;
uint16_t idx = 0;
uint32_t freq = 0;

/**
  * @brief  Initializate pin of tim module.
  * @retval None
  */
void timer_pin_init(void)
{
    ald_gpio_init_t x;

    /* Initialize tim0 ch1 pin */
    x.mode  = ALD_GPIO_MODE_INPUT;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.func  = ALD_GPIO_FUNC_2;
    ald_gpio_init(TIMER_INPUTCAPTURE_PORT, TIMER_INPUTCAPTURE_PIN, &x);
}

/**
  * @brief  Input capture callback  function
  * @retval None
  */
void ald_timer_ic_capturecallback(struct timer_handle_s *htim)
{
    if (htim->ch != ALD_TIMER_ACTIVE_CHANNEL_1)
        return;

    if (idx == 0)
    {
        /* Get the 1st Input Capture value */
        value1 = ald_timer_read_capture_value(htim, ALD_TIMER_CHANNEL_1);
        idx  = 1;

        return;
    }

    if (idx != 1)
        return;

    /* Get the 1st Input Capture value */
    value2 = ald_timer_read_capture_value(htim, ALD_TIMER_CHANNEL_1);

    /* Capture computation */
    if (value2 > value1)
    {
        diff = (value2 - value1);
    }
    else if (value2 < value1)
    {
        /* 0xFFFF is max TIM0_CCRx value */
        diff = ((0xFFFF - value1) + value2) + 1;
    }
    else
    {
        /* If capture values are equal, we have reached the limit of frequency
         * measures
         */
        while (1);
    }

    /* Frequency computation: for this example TIMx (TIM0) is clocked by
     * SystemCoreClock
     */
    freq = (ald_cmu_get_pclk1_clock()) / diff;
    idx = 0;

    return;
}

/**
  * @brief  Main function
  * @retval Status
  */
int main()
{
    /* Initialize ALD */
    ald_cmu_init();
    ald_cmu_pll_config(ALD_CMU_PLL_INPUT_HRC, ALD_CMU_PLL_OUTPUT_48M);
    ald_cmu_clock_config(ALD_CMU_CLOCK_PLL, 48000000);
    ald_cmu_perh_clock_config(ALD_CMU_PERH_ALL, ENABLE);
    /* Initialize pin and set the default state*/
    timer_pin_init();

    /* Initialize AD16C4T0 */
    h_tim.perh           = AD16C4T0;
    h_tim.init.prescaler = 0;
    h_tim.init.mode      = ALD_TIMER_CNT_MODE_UP;
    h_tim.init.period    = 0xFFFF;
    h_tim.init.clk_div   = ALD_TIMER_CLOCK_DIV1;
    h_tim.init.re_cnt    = 0;
    h_tim.capture_cbk    = ald_timer_ic_capturecallback;
    ald_timer_ic_init(&h_tim);

    /* Initialize clock source */
    clock.source     = ALD_TIMER_SRC_INTER;
    ald_timer_config_clock_source(&h_tim, &clock);

    /* Initialize input capture channel */
    init.polarity = ALD_TIMER_IC_POLARITY_RISE;
    init.sel      = ALD_TIMER_IC_SEL_DIRECT;
    init.psc      = ALD_TIMER_IC_PSC_DIV1;
    init.filter   = 0;
    ald_timer_ic_config_channel(&h_tim, &init, ALD_TIMER_CHANNEL_1);
    ald_mcu_irq_config(AD16C4T0_CC_IRQn, 0, 1, ENABLE);

    /* Enable CC interrupt */
    ald_timer_interrupt_config(&h_tim, ALD_TIMER_IT_CC1, ENABLE);

    /* Start input capture */
    ald_timer_ic_start(&h_tim, ALD_TIMER_CHANNEL_1);
    ald_timer_ic_start_by_it(&h_tim, ALD_TIMER_CHANNEL_1);

    while (1)
    {
        ald_delay_ms(1000);
    }
}
/**
  * @}
  */
/**
  * @}
  */
