/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include <string.h>
#include "main.h"
#include "pic.h"
#include "spi_flash.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
static void uart_init(void);  
 
ald_uart_handle_t h_uart;
ald_spi_handle_t h_spi;

uint8_t g_rx_buf[256];
uint8_t g_rx_complete = 0;
uint32_t g_flash_write_addr = 0;
uint8_t g_key_trig = 0;
uint8_t pic_refresh_cnt = 0;
uint8_t pic_refresh_flag = 0;
uint32_t idreg = 0;
uint32_t i = 0;
enum lcd_refresh_status_t lcd_refresh_status;

/**
  * @brief  UART pin init
  * @retval None.
  */
void uart_pin_init(void)
{
    ald_gpio_init_t x;

    /* Initialize tx pin */
    x.mode  = ALD_GPIO_MODE_OUTPUT;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.func  = ALD_GPIO_FUNC_3;
    ald_gpio_init(GPIOB, GPIO_PIN_10, &x);

    /* Initialize rx pin */
    x.mode  = ALD_GPIO_MODE_INPUT;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.func  = ALD_GPIO_FUNC_3;
    ald_gpio_init(GPIOB, GPIO_PIN_11, &x);
}

/**
  * @brief  KEY LED I/O init.
  * @retval None.
  */
void key_led_init(void)
{
    ald_gpio_init_t x;
    ald_exti_init_t exti;
    
    ald_mcu_irq_config(EXTI2_IRQn, 2, 1, ENABLE);

    /* Initialize GPIO_PIN_2 as input */
    x.mode  = ALD_GPIO_MODE_INPUT;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_CMOS;
    x.func  = ALD_GPIO_FUNC_1;
    ald_gpio_init(GPIOD, GPIO_PIN_2, &x);
    /* LED */
    x.mode  = ALD_GPIO_MODE_OUTPUT;
    ald_gpio_init(GPIOA, GPIO_PIN_15, &x);
    /* External interrupt */
    exti.filter      = ENABLE;
    exti.cks         = ALD_EXTI_FILTER_CLOCK_10K;
    exti.filter_time = 10;
    ald_gpio_exti_init(GPIOD, GPIO_PIN_2, &exti);
    /* Clear interrupt flag */
    ald_gpio_exti_clear_flag_status(GPIO_PIN_2);
    /* Configure interrupt */
    ald_gpio_exti_interrupt_config(GPIO_PIN_2, ALD_EXTI_TRIGGER_TRAILING_EDGE, ENABLE);
}

/**
  * @brief  Receive a message complete.
  * @param  arg: Pointer to ald_uart_handle_t structure.
  * @retval None.
  */
void uart_recv_complete(ald_uart_handle_t *arg)
{
    ald_uart_recv_by_it(&h_uart, g_rx_buf, 256);
    g_rx_complete = 0x1;
}

/**
  * @brief  Occurs error.
  * @param  arg: Pointer to ald_uart_handle_t structure.
  * @retval None.
  */
void uart_error(ald_uart_handle_t *arg)
{
    ald_rmu_reset_periperal(ALD_RMU_PERH_UART0);
    uart_init();
    ald_uart_recv_by_it(&h_uart, g_rx_buf, 256);
}

/**
  * @brief  UART init.
  * @retval None.
  */
void uart_init(void)
{
    /* Initialize uart */
    h_uart.perh             = UART0;
    h_uart.init.baud        = 115200;
    h_uart.init.word_length = ALD_UART_WORD_LENGTH_8B;
    h_uart.init.stop_bits   = ALD_UART_STOP_BITS_1;
    h_uart.init.parity      = ALD_UART_PARITY_NONE;
    h_uart.init.mode        = ALD_UART_MODE_UART;
    h_uart.init.fctl        = ALD_UART_HW_FLOW_CTL_DISABLE;
    h_uart.tx_cplt_cbk      = NULL;
    h_uart.rx_cplt_cbk      = uart_recv_complete;
    h_uart.error_cbk        = uart_error;
    ald_uart_init(&h_uart);
    ald_mcu_irq_config(UART0_IRQn, 0, 1, ENABLE);
    ald_uart_recv_by_it(&h_uart, g_rx_buf, 256);
}

/**
  * @brief  Timer callback function.
  * @retval None.
  */
void ald_timer_period_elapsed_callback(struct timer_handle_s *arg)
{
    pic_refresh_flag = 1;
}

/**
  * @brief  Timer init function.
  * @retval None.
  */
void timer_init(void)
{
    ald_timer_handle_t h_tim;
    ald_timer_clock_config_t tim_clock;
    
    /* Initialize GP16C4T0 */
    h_tim.perh = GP16C4T0;
    h_tim.init.prescaler    = 15;
    h_tim.init.mode         = ALD_TIMER_CNT_MODE_UP;
    h_tim.init.period       = 7200 / PIC_FPS - 1; 
    h_tim.init.clk_div      = ALD_TIMER_CLOCK_DIV1;
    h_tim.init.re_cnt       = 0;
    h_tim.period_elapse_cbk = ald_timer_period_elapsed_callback;
    ald_timer_base_init(&h_tim);

    /* Initialize clock source */
    tim_clock.source = ALD_TIMER_SRC_INTER;
    ald_timer_config_clock_source(&h_tim, &tim_clock);

    /* Enable GP16C4T0 interrupt */
    ald_mcu_irq_config(GP16C4T0_IRQn, 3, 1, ENABLE);
    /* Enable UPDATE interrupt */
    ald_timer_interrupt_config(&h_tim, ALD_TIMER_IT_UPDATE, ENABLE);
    /* Start input pwm from GP16C4T0 channel 1 */
    ald_timer_base_start(&h_tim);
}

/**
  * @brief  External FLASH data transferred to SPI1 initialized via DMA
  * @param  size: data size
  * @retval None.
  */
void flash_to_lcd_dma_init(uint32_t size)
{
    ald_dma_handle_t hperh;
    
    hperh.perh = DMA0;
    ald_dma_config_struct(&hperh);
    ald_dma_interrupt_config(&hperh, ALD_DMA_IT_FLAG_TC, ENABLE);

    hperh.config.src             = (void*)&h_spi.perh->DATA;
    hperh.config.src_data_width  = ALD_DMA_DATA_SIZE_HALFWORD;
    hperh.config.src_inc         = ALD_DMA_DATA_INC_DISABLE;
    hperh.config.dst             = (void*)&h_spi1.perh->DATA;
    hperh.config.dst_data_width  = ALD_DMA_DATA_SIZE_HALFWORD;
    hperh.config.dst_inc         = ALD_DMA_DATA_INC_DISABLE;
    hperh.config.mem_to_mem      = DISABLE;

    hperh.config.size          = size;
    hperh.config.priority      = ALD_DMA_LOW_PRIORITY;
    hperh.config.R_power       = ALD_DMA_R_POWER_1;
    hperh.config.dir           = ALD_DMA_DIR_TO_SRAM;
    hperh.config.circle_mode   = DISABLE;
    
    hperh.config.msel          = ALD_DMA_MSEL_SPI0;
    hperh.config.msigsel       = ALD_DMA_MSIGSEL_SPI_RNR;

    hperh.cplt_tc_cbk          = NULL;
    hperh.cplt_tc_arg          = NULL;
    hperh.config.channel       = 0;
    ald_dma_config(&hperh);
}

/**
  * @brief  Tranmit the upper half frame of the image
  * @param  pic_num: picture number
  * @retval None.
  */
void start_pic_transfer_1(uint8_t pic_num)
{
    uint32_t addr = pic_num * PIC_INC;
    uint32_t tmp = 0;
    UNUSED(tmp);

    FLASH_CS_CLR();
    while ((SPI0->STAT & ALD_SPI_STATUS_TXE) == RESET);    
    SPI0->DATA = ((FLASH_READ << 8) | ((addr >> 16) & 0xFF));
    while (((SPI0->STAT & SPI_STAT_RXNE_MSK) != SPI_STAT_RXNE_MSK));
    tmp = SPI0->DATA;
       
    SPI0->DATA = addr & 0xFFFF;
    while (((SPI0->STAT & SPI_STAT_RXNE_MSK) != SPI_STAT_RXNE_MSK));
    tmp = SPI0->DATA;
       
    SPI0->ICR = 0xFFFFFFFF;
    
    /* CHANNEL ENABLE */
    DMA0->CH[0].CON |= 0x1;
    /* SPI RX DMA ENABLE */
    SPI0->CON2  |= 0x1;
    /* SPI RXO */
    SPI0->CON1  |= (0x1 << 10);
}

/**
  * @brief  Tranmit the lower half frame of the image
  * @param  pic_num: picture number
  * @retval None.
  */
void start_pic_transfer_2(uint8_t pic_num)
{
    uint32_t addr = pic_num * PIC_INC + PIC_SIZE / 2;
    uint32_t tmp = 0;
    UNUSED(tmp);

    FLASH_CS_CLR();
    SPI0->ICR = 0xFFFFFFFF;
    while ((SPI0->STAT & ALD_SPI_STATUS_TXE) == RESET);    
    SPI0->DATA = ((FLASH_READ << 8) | ((addr >> 16) & 0xFF));
    while (((SPI0->STAT & SPI_STAT_RXNE_MSK) != SPI_STAT_RXNE_MSK));
    tmp = SPI0->DATA;
    SPI0->DATA = addr & 0xFFFF;
    while (((SPI0->STAT & SPI_STAT_RXNE_MSK) != SPI_STAT_RXNE_MSK));
    tmp = SPI0->DATA;
          
    /* CHANNEL ENABLE */
    DMA0->CH[0].CON |= 0x1;
    /* SPI RX DMA ENABLE */
    SPI0->CON2  |= 0x1;
    /* SPI RXO */
    SPI0->CON1  |= (0x1 << 10);
}

/**
  * @brief  Switch the SPI data width to 16bit
  * @retval None.
  */
void lcd_spi_switch_to_16bit(void)
{
    LCD_CS_Set();
    ALD_SPI_DISABLE(&h_spi);
    ALD_SPI_DISABLE(&h_spi1);
    
    h_spi.perh->CON1  |= SPI_CON1_FLEN_MSK;
    h_spi1.perh->CON1 |= SPI_CON1_FLEN_MSK;
    
    ALD_SPI_ENABLE(&h_spi);
    ALD_SPI_ENABLE(&h_spi1);
    LCD_CS_Clr();    
}

/**
  * @brief  Switch the SPI data width to 8bit
  * @retval None.
  */
void lcd_spi_switch_to_8bit(void)
{
    LCD_CS_Set();
    ALD_SPI_DISABLE(&h_spi);
    ALD_SPI_DISABLE(&h_spi1);
    
    h_spi.perh->CON1  &=~ SPI_CON1_FLEN_MSK;
    h_spi1.perh->CON1 &=~ SPI_CON1_FLEN_MSK;
    
    ALD_SPI_ENABLE(&h_spi);
    ALD_SPI_ENABLE(&h_spi1);
    LCD_CS_Clr();    
}

/**
  * @brief  Image refresh control
  * @retval None.
  */
void pic_refresh_check(void)
{
    if(lcd_refresh_status == idle && pic_refresh_flag == 1)
    {
        pic_refresh_flag = 0;
        lcd_refresh_status = first_in_transfer;
        
        lcd_spi_switch_to_8bit();
        LCD_Address_Set(0, 0, 320 - 1, 240 - 1);
        lcd_spi_switch_to_16bit();
        
        start_pic_transfer_1(pic_refresh_cnt);
    }
    if(lcd_refresh_status == first_complete)
    {
        lcd_refresh_status = second_in_transfer;
        start_pic_transfer_2(pic_refresh_cnt);
    }
    if(lcd_refresh_status == second_complete)
    {
        lcd_refresh_status = idle;
        pic_refresh_cnt++;
        if(pic_refresh_cnt == PIC_NUM)
        {
            pic_refresh_cnt = 0;
        }
    }
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    ald_cmu_init();
    /* Configure system clock */
    ald_cmu_pll_config(ALD_CMU_PLL_INPUT_HRC, ALD_CMU_PLL_OUTPUT_72M);
    ald_cmu_clock_config(ALD_CMU_CLOCK_PLL, 72000000);
    ald_cmu_perh_clock_config(ALD_CMU_PERH_ALL, ENABLE);
    memset(g_rx_buf, 0x55, sizeof(g_rx_buf));
    
    key_led_init();
    uart_pin_init();
    uart_init();
    flash_module_init();
    idreg = flash_read_id();
    
    /* Waiting for key press */
    while(g_key_trig == 0)
    {
        /* Waiting for receive data */
        if(g_rx_complete)
        {
            g_rx_complete = 0;
            if((g_flash_write_addr % 0x1000) == 0)
            {
                flash_sector_erase(g_flash_write_addr);
                flash_wait_unbusy();
            }
            flash_write(g_flash_write_addr, g_rx_buf, 256);
            flash_wait_unbusy();
            g_flash_write_addr += 256;
            UART0->TXBUF = 0X55;
        }
    } 
    g_key_trig = 0;  

    LCD_CS_Clr();
    LCD_Init();
    LCD_Fill(0, 0, 319, 239, WHITE);
    
    flash_to_lcd_dma_init(PIC_SIZE / 4);
    lcd_spi_switch_to_16bit();
    timer_init();
    lcd_refresh_status = idle;
    while(1)
    {
        pic_refresh_check();
    }
}




/**
  * @}
  */
/**
  * @}
  */
