#include "lcd.h"
#include "main.h"
#include "lcd_font.h"


ald_spi_handle_t h_spi1;

/***********缓冲区适合写入大量数据******/
// 定义发送缓冲区的最大大小
#define LCD_BUF_MAXSIZE (Width*Height*2)
// 定义发送的次数
#define SPI_SendCount   100
// 单次发送的缓冲区大小
#define LCD_BUF_SIZE    (LCD_BUF_MAXSIZE/SPI_SendCount)

/* ST7789 specific commands used in init*/
#define ST7789_NOP          0x00
#define ST7789_SWRESET      0x01
#define ST7789_RDDID        0x04
#define ST7789_RDDST        0x09

#define ST7789_RDDPM        0x0A      /*Read display power mode*/
#define ST7789_RDD_MADCTL   0x0B      /*Read display MADCTL*/
#define ST7789_RDD_COLMOD   0x0C      /*Read display pixel format*/
#define ST7789_RDDIM        0x0D      /*Read display image mode*/
#define ST7789_RDDSM        0x0E      /*Read display signal mode*/
#define ST7789_RDDSR        0x0F      /*Read display self-diagnostic result (ST7789V)*/

#define ST7789_SLPIN        0x10
#define ST7789_SLPOUT       0x11
#define ST7789_PTLON        0x12
#define ST7789_NORON        0x13

#define ST7789_INVOFF       0x20
#define ST7789_INVON        0x21
#define ST7789_GAMSET       0x26      /*Gamma set*/
#define ST7789_DISPOFF      0x28
#define ST7789_DISPON       0x29
#define ST7789_CASET        0x2A
#define ST7789_RASET        0x2B
#define ST7789_RAMWR        0x2C
#define ST7789_RGBSET       0x2D      /*Color setting for 4096, 64K and 262K colors*/
#define ST7789_RAMRD        0x2E

#define ST7789_PTLAR        0x30
#define ST7789_VSCRDEF      0x33      /*Vertical scrolling definition (ST7789V)*/
#define ST7789_TEOFF        0x34      /*Tearing effect line off*/
#define ST7789_TEON         0x35      /*Tearing effect line on*/
#define ST7789_MADCTL       0x36      /*Memory data access control*/
#define ST7789_IDMOFF       0x38      /*Idle mode off*/
#define ST7789_IDMON        0x39      /*Idle mode on*/
#define ST7789_RAMWRC       0x3C      /*Memory write continue (ST7789V)*/
#define ST7789_RAMRDC       0x3E      /*Memory read continue (ST7789V)*/
#define ST7789_COLMOD       0x3A

#define ST7789_RAMCTRL      0xB0      /*RAM control*/
#define ST7789_RGBCTRL      0xB1      /*RGB control*/
#define ST7789_PORCTRL      0xB2      /*Porch control*/
#define ST7789_FRCTRL1      0xB3      /*Frame rate control*/
#define ST7789_PARCTRL      0xB5      /*Partial mode control*/
#define ST7789_GCTRL        0xB7      /*Gate control*/
#define ST7789_GTADJ        0xB8      /*Gate on timing adjustment*/
#define ST7789_DGMEN        0xBA      /*Digital gamma enable*/
#define ST7789_VCOMS        0xBB      /*VCOMS setting*/
#define ST7789_LCMCTRL      0xC0      /*LCM control*/
#define ST7789_IDSET        0xC1      /*ID setting*/
#define ST7789_VDVVRHEN     0xC2      /*VDV and VRH command enable*/
#define ST7789_VRHS         0xC3      /*VRH set*/
#define ST7789_VDVSET       0xC4      /*VDV setting*/
#define ST7789_VCMOFSET     0xC5      /*VCOMS offset set*/
#define ST7789_FRCTR2       0xC6      /*FR Control 2*/
#define ST7789_CABCCTRL     0xC7      /*CABC control*/
#define ST7789_REGSEL1      0xC8      /*Register value section 1*/
#define ST7789_REGSEL2      0xCA      /*Register value section 2*/
#define ST7789_PWMFRSEL     0xCC      /*PWM frequency selection*/
#define ST7789_PWCTRL1      0xD0      /*Power control 1*/
#define ST7789_VAPVANEN     0xD2      /*Enable VAP/VAN signal output*/
#define ST7789_CMD2EN       0xDF      /*Command 2 enable*/
#define ST7789_PVGAMCTRL    0xE0      /*Positive voltage gamma control*/
#define ST7789_NVGAMCTRL    0xE1      /*Negative voltage gamma control*/
#define ST7789_DGMLUTR      0xE2      /*Digital gamma look-up table for red*/
#define ST7789_DGMLUTB      0xE3      /*Digital gamma look-up table for blue*/
#define ST7789_GATECTRL     0xE4      /*Gate control*/
#define ST7789_SPI2EN       0xE7      /*SPI2 enable*/
#define ST7789_PWCTRL2      0xE8      /*Power control 2*/
#define ST7789_EQCTRL       0xE9      /*Equalize time control*/
#define ST7789_PROMCTRL     0xEC      /*Program control*/
#define ST7789_PROMEN       0xFA      /*Program mode enable*/
#define ST7789_NVMSET       0xFC      /*NVM setting*/
#define ST7789_PROMACT      0xFE      /*Program action*/

#define ES_LCD_X2Y_MASK             0x20
#define ES_LCD_X2Y             		ES_LCD_X2Y_MASK

#define ES_LCD_X_ORDER 				0U

#define ES_LCD_Y_ORDER_MASK 		0x80
#define ES_LCD_Y_ORDER 				ES_LCD_Y_ORDER_MASK

/**
  * @brief  Initializate LCD moudle
  * @retval None.
  */
static void lcd_spi_init(void)
{
    memset(&h_spi1, 0, sizeof(h_spi1));

    h_spi1.perh           = SPI1;
    h_spi1.init.mode      = ALD_SPI_MODE_MASTER;
    h_spi1.init.dir       = ALD_SPI_DIRECTION_2LINES;
    h_spi1.init.data_size = ALD_SPI_DATA_SIZE_8;
    h_spi1.init.baud      = ALD_SPI_BAUD_2;
    h_spi1.init.phase     = ALD_SPI_CPHA_SECOND;
    h_spi1.init.polarity  = ALD_SPI_CPOL_HIGH;
    h_spi1.init.first_bit = ALD_SPI_FIRSTBIT_MSB;
    h_spi1.init.ss_en     = DISABLE;
    h_spi1.init.crc_calc  = DISABLE;

    ald_spi_init(&h_spi1);
}

/**
  * @brief  Initializate lcd pin
  * @retval None.
  */
static void LCD_GPIO_Init(void)
{
    ald_gpio_init_t x;

    /* Initialize nss pin */
    x.mode  = ALD_GPIO_MODE_OUTPUT;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.func  = ALD_GPIO_FUNC_1;
    ald_gpio_init(GPIOB, GPIO_PIN_12, &x);
    /* IO_BK*/
    ald_gpio_init(GPIOA, GPIO_PIN_10, &x);
    /* LCD_RST */
    ald_gpio_init(GPIOA, GPIO_PIN_9, &x);
    /* LCD_WR */
    ald_gpio_init(GPIOC, GPIO_PIN_8, &x);
    /* Initialize sck pin */
    x.mode  = ALD_GPIO_MODE_OUTPUT;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.func  = ALD_GPIO_FUNC_4;
    ald_gpio_init(GPIOB, GPIO_PIN_13, &x);
    /* Initialize miso pin */
    x.mode  = ALD_GPIO_MODE_INPUT;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.func  = ALD_GPIO_FUNC_4;
    x.func  = ALD_GPIO_FUNC_4;
    ald_gpio_init(GPIOB, GPIO_PIN_14, &x);
    /* Initialize mosi pin */
    x.mode  = ALD_GPIO_MODE_OUTPUT;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.func  = ALD_GPIO_FUNC_4;
    x.func  = ALD_GPIO_FUNC_4;
    ald_gpio_init(GPIOB, GPIO_PIN_15, &x);
   
    lcd_spi_init();
}


/******************************************************************************
      函数说明：LCD串行数据写入函数
      入口数据：dat  要写入的串行数据
      返回值：  无
******************************************************************************/
void LCD_Writ_Bus(uint8_t dat) 
{
    ald_spi_send_byte_fast_1line(&h_spi1, dat);
}

/******************************************************************************
      函数说明：LCD写入数据
      入口数据：dat 写入的数据
      返回值：  无
******************************************************************************/
void LCD_WR_DATA8(uint8_t dat)
{
    LCD_Writ_Bus(dat);
}

/******************************************************************************
      函数说明：LCD写入数据
      入口数据：dat 写入的数据
      返回值：  无
******************************************************************************/
void LCD_WR_DATA(uint16_t dat)
{
    LCD_Writ_Bus(dat>>8);
    LCD_Writ_Bus(dat);
}

/******************************************************************************
      函数说明：LCD写入命令
      入口数据：dat 写入的命令
      返回值：  无
******************************************************************************/
void LCD_WR_REG(uint8_t dat)
{
    LCD_DC_Clr();//写命令
    LCD_Writ_Bus(dat);
    LCD_DC_Set();//写数据
}

/******************************************************************************
      函数说明：设置起始和结束地址
      入口数据：x1,x2 设置列的起始和结束地址
                y1,y2 设置行的起始和结束地址
      返回值：  无
******************************************************************************/
void LCD_Address_Set(uint16_t x1,uint16_t y1,uint16_t x2,uint16_t y2)
{
	LCD_WR_REG(0x2A);
	LCD_Writ_Bus(x1>>8);
	LCD_Writ_Bus(x1);
	LCD_Writ_Bus(x2>>8);
	LCD_Writ_Bus(x2);
	LCD_WR_REG(0x2B);
	LCD_Writ_Bus(y1>>8);
	LCD_Writ_Bus(y1);
	LCD_Writ_Bus(y2>>8);
	LCD_Writ_Bus(y2);
	LCD_WR_REG(0x2C);
}

/******************************************************************************
      函数说明：在指定区域填充颜色
      入口数据：xsta,ysta   起始坐标
                xend,yend   终止坐标
                color       要填充的颜色
      返回值：  无
******************************************************************************/
void LCD_Fill(uint16_t xsta,uint16_t ysta,uint16_t xend,uint16_t yend,uint16_t color)
{          
	uint16_t i,j; 
	LCD_Address_Set(xsta,ysta,xend-1,yend-1);//设置显示范围
	for(i=ysta;i<yend;i++)
	{   
		for(j=xsta;j<xend;j++)
		{
			LCD_WR_DATA(color);
		}
	} 					  	    
}

/******************************************************************************
      函数说明：在指定位置画点
      入口数据：x,y 画点坐标
                color 点的颜色
      返回值：  无
******************************************************************************/
void LCD_DrawPoint(uint16_t x,uint16_t y,uint16_t color)
{
	LCD_Address_Set(x,y,x,y);//设置光标位置 
	LCD_WR_DATA(color);
} 

/******************************************************************************
      函数说明：画线
      入口数据：x1,y1   起始坐标
                x2,y2   终止坐标
                color   线的颜色
      返回值：  无
******************************************************************************/
void LCD_DrawLine(uint16_t x1,uint16_t y1,uint16_t x2,uint16_t y2,uint16_t color)
{
	uint16_t t; 
	int xerr=0,yerr=0,delta_x,delta_y,distance;
	int incx,incy,uRow,uCol;
	delta_x=x2-x1; //计算坐标增量 
	delta_y=y2-y1;
	uRow=x1;//画线起点坐标
	uCol=y1;
	if(delta_x>0)incx=1; //设置单步方向 
	else if (delta_x==0)incx=0;//垂直线 
	else {incx=-1;delta_x=-delta_x;}
	if(delta_y>0)incy=1;
	else if (delta_y==0)incy=0;//水平线 
	else {incy=-1;delta_y=-delta_y;}
	if(delta_x>delta_y)distance=delta_x; //选取基本增量坐标轴 
	else distance=delta_y;
	for(t=0;t<distance+1;t++)
	{
		LCD_DrawPoint(uRow,uCol,color);//画点
		xerr+=delta_x;
		yerr+=delta_y;
		if(xerr>distance)
		{
			xerr-=distance;
			uRow+=incx;
		}
		if(yerr>distance)
		{
			yerr-=distance;
			uCol+=incy;
		}
	}
}


/******************************************************************************
      函数说明：画矩形
      入口数据：x1,y1   起始坐标
                x2,y2   终止坐标
                color   矩形的颜色
      返回值：  无
******************************************************************************/
void LCD_DrawRectangle(uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2,uint16_t color)
{
	LCD_DrawLine(x1,y1,x2,y1,color);
	LCD_DrawLine(x1,y1,x1,y2,color);
	LCD_DrawLine(x1,y2,x2,y2,color);
	LCD_DrawLine(x2,y1,x2,y2,color);
}


/******************************************************************************
      函数说明：画圆
      入口数据：x0,y0   圆心坐标
                r       半径
                color   圆的颜色
      返回值：  无
******************************************************************************/
void Draw_Circle(uint16_t x0,uint16_t y0,uint8_t r,uint16_t color)
{
	int a,b;
	a=0;b=r;	  
	while(a<=b)
	{
		LCD_DrawPoint(x0-b,y0-a,color);             //3           
		LCD_DrawPoint(x0+b,y0-a,color);             //0           
		LCD_DrawPoint(x0-a,y0+b,color);             //1                
		LCD_DrawPoint(x0-a,y0-b,color);             //2             
		LCD_DrawPoint(x0+b,y0+a,color);             //4               
		LCD_DrawPoint(x0+a,y0-b,color);             //5
		LCD_DrawPoint(x0+a,y0+b,color);             //6 
		LCD_DrawPoint(x0-b,y0+a,color);             //7
		a++;
		if((a*a+b*b)>(r*r))//判断要画的点是否过远
		{
			b--;
		}
	}
}

/******************************************************************************
      函数说明：显示图片
      入口数据：x,y起点坐标
                length 图片长度
                width  图片宽度
                pic[]  图片数组    
      返回值：  无
******************************************************************************/
void LCD_ShowPicture(uint16_t x,uint16_t y,uint16_t length,uint16_t width,const uint8_t pic[])
{
    uint16_t i,j;
    uint32_t k=0;
    LCD_Address_Set(x,y,x+length-1,y+width-1);
    for(i=0;i<length;i++)
    {
        for(j=0;j<width;j++)
        {
            LCD_WR_DATA8(pic[k*2]);
            LCD_WR_DATA8(pic[k*2+1]);
            k++;
        }
    }   
}

void LCD_Init(void)
{
    LCD_BLK_Set();
    LCD_GPIO_Init();
#if 1    
    LCD_RES_Set();
    ald_delay_ms(100);
    LCD_RES_Clr();
    ald_delay_ms(200);
    LCD_RES_Set();
    ald_delay_ms(500);

    //************* Start Initial Sequence **********//
    LCD_WR_REG(0x11); //Sleep out 
    ald_delay_ms(120);//Delay 120ms

    LCD_WR_REG(ST7789_NORON);    /* Normal display mode on */

    //************* Start Initial Sequence **********//

    LCD_WR_REG(ST7789_MADCTL);
    LCD_WR_DATA8(ES_LCD_X2Y | ES_LCD_X_ORDER | ES_LCD_Y_ORDER);

    /* JLX240 display datasheet */
    LCD_WR_REG(0xB6);
    LCD_WR_DATA8(0x0A);
    LCD_WR_DATA8(0x82);

    LCD_WR_REG(ST7789_RAMCTRL);
    LCD_WR_DATA8(0x00);
    LCD_WR_DATA8(0xE0); /* 5 to 6 bit conversion: r0 = r5, b0 = b5*/

    LCD_WR_REG(ST7789_COLMOD);
    LCD_WR_DATA8(0x55);
    ald_delay_ms(10);

    /*--------------------------------ST7789V Frame rate setting----------------------------------*/
    LCD_WR_REG(ST7789_PORCTRL);
    LCD_WR_DATA8(0x0c);
    LCD_WR_DATA8(0x0c);
    LCD_WR_DATA8(0x00);
    LCD_WR_DATA8(0x33);
    LCD_WR_DATA8(0x33);

    LCD_WR_REG(ST7789_GCTRL);      /* Voltages: VGH / VGL*/
    LCD_WR_DATA8(0x35);

    /*---------------------------------ST7789V Power setting--------------------------------------*/
    LCD_WR_REG(ST7789_VCOMS);
    LCD_WR_DATA8(0x28);       /* JLX240 display datasheet*/

    LCD_WR_REG(ST7789_LCMCTRL);
    LCD_WR_DATA8(0x2C);

    LCD_WR_REG(ST7789_VDVVRHEN);
    LCD_WR_DATA8(0x01);
    LCD_WR_DATA8(0xFF);

    LCD_WR_REG(ST7789_VRHS);       /* voltage VRHS*/
    LCD_WR_DATA8(0x10);

    LCD_WR_REG(ST7789_VDVSET);
    LCD_WR_DATA8(0x20);

    LCD_WR_REG(ST7789_FRCTR2);
    LCD_WR_DATA8(0x0f);

    LCD_WR_REG(ST7789_PWCTRL1);
    LCD_WR_DATA8(0xa4);
    LCD_WR_DATA8(0xa1);

    /*--------------------------------ST7789V gamma setting---------------------------------------*/
    LCD_WR_REG(ST7789_PVGAMCTRL);
    LCD_WR_DATA8(0xd0);
    LCD_WR_DATA8(0x00);
    LCD_WR_DATA8(0x02);
    LCD_WR_DATA8(0x07);
    LCD_WR_DATA8(0x0a);
    LCD_WR_DATA8(0x28);
    LCD_WR_DATA8(0x32);
    LCD_WR_DATA8(0x44);
    LCD_WR_DATA8(0x42);
    LCD_WR_DATA8(0x06);
    LCD_WR_DATA8(0x0e);
    LCD_WR_DATA8(0x12);
    LCD_WR_DATA8(0x14);
    LCD_WR_DATA8(0x17);

    LCD_WR_REG(ST7789_NVGAMCTRL);
    LCD_WR_DATA8(0xd0);
    LCD_WR_DATA8(0x00);
    LCD_WR_DATA8(0x02);
    LCD_WR_DATA8(0x07);
    LCD_WR_DATA8(0x0a);
    LCD_WR_DATA8(0x28);
    LCD_WR_DATA8(0x31);
    LCD_WR_DATA8(0x54);
    LCD_WR_DATA8(0x47);
    LCD_WR_DATA8(0x0e);
    LCD_WR_DATA8(0x1c);
    LCD_WR_DATA8(0x17);
    LCD_WR_DATA8(0x1b);
    LCD_WR_DATA8(0x1e);

    LCD_WR_REG(ST7789_INVON);

    LCD_WR_REG(ST7789_DISPON);    /* Display on */

    ald_delay_ms(120);
    #endif
} 


// LCD打开电源
void LCD_PWR_ON(void)
{
    LCD_BLK_Set();
}

// LCD关闭电源
void LCD_PWR_OFF(void)
{
    LCD_BLK_Clr();
}
























