/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include <string.h>
#include "main.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

uint8_t buf_tx[255];
uint8_t buf_rx[255];
static ald_spi_handle_t h_spi;
/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    /* Initialize ALD */
    ald_cmu_init();
    /* Configure system clock */
    ald_cmu_pll_config(ALD_CMU_PLL_INPUT_HRC, ALD_CMU_PLL_OUTPUT_48M);
    ald_cmu_clock_config(ALD_CMU_CLOCK_PLL, 48000000);
    ald_cmu_perh_clock_config(ALD_CMU_PERH_ALL, ENABLE);

    memset(buf_tx, 0x55, sizeof(buf_tx));

    /* Initialize w25q32 */
    w25q32_module_init();

    meter_bsp_spi_flash_read_id();
    ald_delay_ms(10);
    w25q32_sector_erase(0);
    ald_delay_ms(100);
    w25q32_write(0, buf_tx, sizeof(buf_tx));
    ald_delay_ms(10);
    w25q32_read(0, buf_rx, sizeof(buf_rx));

    while (1)
    {
        ald_delay_ms(1000);
    }
}

/**
  * @brief  Initializate spi_flash pin
  * @retval None.
  */
static void w25q32_pin_init(void)
{
    ald_gpio_init_t x;

    /* Initialize nss pin */
    x.mode  = ALD_GPIO_MODE_OUTPUT;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.func  = ALD_GPIO_FUNC_1;
    ald_gpio_init(GPIOA, GPIO_PIN_4, &x);
	ald_gpio_write_pin(GPIOA, GPIO_PIN_4, 1);

    /* Initialize sck pin */
    x.mode  = ALD_GPIO_MODE_OUTPUT;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.func  = ALD_GPIO_FUNC_6;
    ald_gpio_init(GPIOA, GPIO_PIN_5, &x);

    /* Initialize miso pin */
    x.mode  = ALD_GPIO_MODE_INPUT;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.func  = ALD_GPIO_FUNC_6;
    ald_gpio_init(GPIOA, GPIO_PIN_6, &x);

    /* Initialize mosi pin */
    x.mode  = ALD_GPIO_MODE_OUTPUT;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.func  = ALD_GPIO_FUNC_6;
    ald_gpio_init(GPIOA, GPIO_PIN_7, &x);
}

/**
  * @brief  Initializate spi tx delay
  * @retval None.
  */
static void __delay(void)
{
    int i = 100;

    while (i--);
}

/**
  * @brief  Initializate w25q32 moudle
  * @retval None.
  */
void w25q32_module_init(void)
{
    memset(&h_spi, 0, sizeof(h_spi));

    w25q32_pin_init();

    h_spi.perh           = SPI0;
    h_spi.init.mode      = ALD_SPI_MODE_MASTER;
    h_spi.init.dir       = ALD_SPI_DIRECTION_2LINES;
    h_spi.init.data_size = ALD_SPI_DATA_SIZE_8;
    h_spi.init.baud      = ALD_SPI_BAUD_8;
    h_spi.init.phase     = ALD_SPI_CPHA_SECOND;
    h_spi.init.polarity  = ALD_SPI_CPOL_HIGH;
    h_spi.init.first_bit = ALD_SPI_FIRSTBIT_MSB;
    h_spi.init.ss_en     = DISABLE;
    h_spi.init.crc_calc  = DISABLE;

    ald_spi_init(&h_spi);
}

/**
  * @brief  transmit an amount of data in blocking mode.
  * @param  addr: Specific address which to be write.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be sent
  * @retval Status, see @ref w25_status_t.
  */
w25_status_t w25q32_write(uint32_t addr, uint8_t *buf, uint16_t size)
{
    uint8_t write_cmd  = 0x02;
    uint8_t write_enable_cmd = 0x06;
    uint8_t tx_buf[4] = {0};
    uint16_t i = 0;

    if (buf == NULL)
        return W25_ERROR;

    tx_buf[0] = write_cmd;
    tx_buf[1] = (addr >> 16) & 0xff;
    tx_buf[2] = (addr >> 8) & 0xff;
    tx_buf[3] = addr & 0xff;

    W25Q32_CS_CLR();

    if (ald_spi_send_byte_fast(&h_spi, write_enable_cmd) != ALD_OK)
    {
        W25Q32_CS_SET();
        return W25_ERROR;
    }

    __delay();
    W25Q32_CS_SET();
    __delay();
    W25Q32_CS_CLR();

    for (i = 0; i < 4; i++)
    {
        if (ald_spi_send_byte_fast(&h_spi, tx_buf[i]) != ALD_OK)
        {
            W25Q32_CS_SET();
            return W25_ERROR;
        }
    }

    for (i = 0; i < size; i++)
    {
        if (ald_spi_send_byte_fast(&h_spi, buf[i]) != ALD_OK)
        {
            W25Q32_CS_SET();
            return W25_ERROR;
        }
    }

    __delay();
    W25Q32_CS_SET();
    return W25_OK;
}

/**
  * @brief  Receive an amount of data in blocking mode with fast read mode.
  * @param  addr: address of flash where want to read.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be received
  * @retval Status, see @ref w25_status_t.
  */
w25_status_t w25q32_fast_read(uint32_t addr, uint8_t *buf, uint16_t size)
{
    uint8_t read_cmd = 0x0B;
    uint8_t add_buf[5] = {0};
    uint16_t i = 0;
    int r_flag = 0;

    W25Q32_CS_CLR();

    add_buf[0] = read_cmd;
    add_buf[1] = (addr >> 16) & 0xff;
    add_buf[2] = (addr >> 8) & 0xff;
    add_buf[3] = addr & 0xff;

    if (buf == NULL)
        return W25_ERROR;

    for (i = 0; i < 4; i++)
    {
        if (ald_spi_send_byte_fast(&h_spi, add_buf[i]) != ALD_OK)
        {
            W25Q32_CS_SET();
            return W25_ERROR;
        }
    }

    ald_spi_recv_byte_fast(&h_spi, &r_flag);

    for (i = 0; i < size; i++)
    {
        buf[i] = ald_spi_recv_byte_fast(&h_spi, &r_flag);

        if (r_flag != ALD_OK)
        {
            W25Q32_CS_SET();
            return W25_ERROR;
        }
    }

    W25Q32_CS_SET();

    return W25_OK;
}

/**
  * @brief  Receive an amount of data in blocking mode.
  * @param  addr: address of flash where want to read.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be received
  * @retval Status, see @ref w25_status_t.
  */
w25_status_t w25q32_read(uint32_t addr, uint8_t *buf, uint16_t size)
{
    uint8_t read_cmd = 0x03;
    uint8_t add_buf[5] = {0};
    uint16_t i = 0;
    int r_flag = 0;

    W25Q32_CS_CLR();

    add_buf[0] = read_cmd;
    add_buf[1] = (addr >> 16) & 0xff;
    add_buf[2] = (addr >> 8) & 0xff;
    add_buf[3] = addr & 0xff;

    if (buf == NULL)
        return W25_ERROR;

    for (i = 0; i < 4; i++)
    {
        if (ald_spi_send_byte_fast(&h_spi, add_buf[i]) != ALD_OK)
        {
            W25Q32_CS_SET();
            return W25_ERROR;
        }
    }

    for (i = 0; i < size; i++)
    {
        buf[i] = ald_spi_recv_byte_fast(&h_spi, &r_flag);

        if (r_flag != ALD_OK)
        {
            W25Q32_CS_SET();
            return W25_ERROR;
        }
    }

    __delay();
    W25Q32_CS_SET();

    return W25_OK;
}
/**
  * @brief  Erase an sector of data in blocking mode.
  * @param  addr: specific sector address want erase.
  * @retval Status, see @ref w25_status_t.
  */
w25_status_t w25q32_sector_erase(uint8_t addr)
{
    uint8_t sector_erase_cmd = 0x20;
    uint8_t write_enable_cmd = 0x06;
    uint8_t add_buf[4] = {0};
    uint8_t i = 0;

    add_buf[0] = sector_erase_cmd;
    add_buf[1] = (addr >> 16) & 0xff;
    add_buf[2] = (addr >> 8) & 0xff;
    add_buf[3] = addr & 0xff;

    W25Q32_CS_CLR();

    if (ald_spi_send_byte_fast(&h_spi, write_enable_cmd) != ALD_OK)
    {
        W25Q32_CS_SET();
        return W25_ERROR;
    }

    __delay();
    W25Q32_CS_SET();
    __delay();
    W25Q32_CS_CLR();

    for (i = 0; i < 4; i++)
    {
        if (ald_spi_send_byte_fast(&h_spi, add_buf[i]) != ALD_OK)
        {
            W25Q32_CS_SET();
            return W25_ERROR;
        }
    }

    __delay();
    W25Q32_CS_SET();

    return W25_OK;
}


/**
  * @brief  Read w25q32 id in blocking mode.
  * @retval w25q32 id.
  */
uint32_t meter_bsp_spi_flash_read_id(void)
{
    uint8_t i;
    int r_flag = 0;
    uint8_t flash_id[4] = {0};

    flash_id[0] = 0x90;

    W25Q32_CS_CLR();

    for (i = 0; i < 4; i++)
    {
        if (ald_spi_send_byte_fast(&h_spi, flash_id[i]) != ALD_OK)
        {
            W25Q32_CS_SET();
            return W25_ERROR;
        }
    }

    __delay();

    for (i = 0; i < 2; i++)
    {
        flash_id[i] = ald_spi_recv_byte_fast(&h_spi, &r_flag);

        if (r_flag != ALD_OK)
        {
            W25Q32_CS_SET();
            return W25_ERROR;
        }
    }

    __delay();
    W25Q32_CS_SET();

    return ((flash_id[0] << 24) | (flash_id[1] << 16) | (flash_id[2] << 8) | (flash_id[3]));
}
/**
  * @}
  */
/**
  * @}
  */
