/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    12 Mar 2023
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2023     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include <string.h>
#include "main.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

ald_rtc_time_t time;
ald_rtc_date_t date;
ald_rtc_alarm_t alarm;
uint8_t temp[256];
uint32_t idx = 0;

void gpio_init(void)
{
    ald_gpio_init_t x;
    /* Initialize GPIO_PIN_15 as output */
    x.mode  = ALD_GPIO_MODE_OUTPUT;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_CMOS;
    x.func  = ALD_GPIO_FUNC_1;
    ald_gpio_init(GPIOC, GPIO_PIN_12, &x);
}


/**
  * @brief  Test RTC time and date functions
  * @retval Status.
  */
int main()
{
    ald_rtc_init_t init;

    /* Initialize ALD */
    ald_cmu_init();
    /* Configure system clock */
    ald_cmu_pll_config(ALD_CMU_PLL_INPUT_HRC, ALD_CMU_PLL_OUTPUT_48M);
    ald_cmu_clock_config(ALD_CMU_CLOCK_PLL, 48000000);
    /* Set PCLK2 frequency */
    ald_cmu_div_config(ALD_CMU_PCLK_2, ALD_CMU_DIV_4);
    /* Enable all peripherals */
    ald_cmu_perh_clock_config(ALD_CMU_PERH_ALL, ENABLE);
    /* Enable RTC interrupt */
    ald_mcu_irq_config(RTC_IRQn, 0, 3, ENABLE);
    
    gpio_init();

    /* Initialize RTC */
    init.asynch_pre_div  = 0;
    init.synch_pre_div   = 32767;
    init.hour_format     = ALD_RTC_HOUR_FORMAT_24;
    init.output          = ALD_RTC_OUTPUT_DISABLE;
    init.output_polarity = ALD_RTC_OUTPUT_POLARITY_HIGH;
    ald_rtc_init(&init);

    /* Set current time and date */
    time.hour    = 21;
    time.minute  = 22;
    time.second  = 38;
    date.day     = 11;
    date.month   = 9;
    date.year    = 24;
    date.week    = 1;
    ald_rtc_set_time(&time, ALD_RTC_FORMAT_DEC);
    ald_rtc_set_date(&date, ALD_RTC_FORMAT_DEC);

    ald_rtc_set_wakeup(ALD_RTC_WAKEUP_CLOCK_DIV_2, (32768 >> 2) - 1);
    ald_rtc_interrupt_config(ALD_RTC_IT_WU, ENABLE);
    ald_pmu_stop1_enter();

    while (1)
    {
        ald_gpio_toggle_pin(GPIOC, GPIO_PIN_12);
        ald_delay_ms(500);
    }
}

/**
  * @brief  RTC IRQ handler
  * @retval None
  */
void rtc_irq_handler(void)
{
    ald_rtc_get_time(&time, ALD_RTC_FORMAT_DEC);
    temp[idx++] = time.second;
    idx = idx == 256 ? 0 : idx;

    ald_rtc_get_flag_status(ALD_RTC_IF_WU);
    ald_rtc_clear_flag_status(ALD_RTC_IF_WU);

    return;
}
/**
  * @}
  */
/**
  * @}
  */
