/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include "main.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples_PMU
  * @{
  */

#define  BKPC_VALUE_W   0x55555555

ald_uart_handle_t h_uart;
char uart_buf[64];
uint32_t bkpc_value_read  = 0;

uint8_t rx_buf[3];
volatile uint8_t command = 0;

void uart_stdio_init(void);
void printf_e(const char *fmt, ...);
void exti_pin_init(void);
void system_check(void);

/***********************************************************command***************************************************************
* FA 01 FA ---------- enter stop1 mode
* FA 02 FA ---------- enter stop2 mode
* FA 03 FA ---------- enter standby mode,  waked by PA1(high)
* FA 04 FA ---------- enter standby mode,  waked by PA1(high) and RTC MIN interrupt
* FA 05 FA ---------- RTC initialization
* FA 06 FA ---------- write data (0x55555555) to backups RAM
* FA 07 FA ---------- read data from backups RAM
* FA 08 FA ---------- read RTC->CON & RTC->PSR
* FA 09 FA ---------- read PMU->SR  & RMU->RSTSR
**********************************************************************************************************************************/

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    ald_rtc_init_t init;
    ald_rtc_time_t time;
    ald_rtc_date_t date;
    uint32_t i = 0;
    ald_exti_init_t exti;

    /* Initialize ALD */
    ald_cmu_init();
    ald_cmu_pll_config(ALD_CMU_PLL_INPUT_HRC, ALD_CMU_PLL_OUTPUT_48M);
    ald_cmu_clock_config(ALD_CMU_CLOCK_PLL, 48000000);
    ald_cmu_perh_clock_config(ALD_CMU_PERH_ALL, ENABLE);
   
    exti_pin_init();

    ald_mcu_irq_config(EXTI1_IRQn, 2, 2, ENABLE);
    ald_mcu_irq_config(RTC_IRQn, 2, 2, ENABLE);
    ald_mcu_irq_config(UART0_IRQn, 1, 0, ENABLE);

    uart_stdio_init();
    printf_e("System start...\r\n");

    system_check();
    command = 0;

    /* Initialize external interrupt */
    exti.filter      = ENABLE;
    exti.cks         = ALD_EXTI_FILTER_CLOCK_10K;
    exti.filter_time = 10;
    ald_gpio_exti_init(PMU_EXTI_PORT, PMU_EXTI_PIN, &exti);

    /* Clear interrupt flag */
    ald_gpio_exti_clear_flag_status(PMU_EXTI_PIN);
    /* Configure interrupt */
    ald_gpio_exti_interrupt_config(PMU_EXTI_PIN, ALD_EXTI_TRIGGER_RISING_EDGE, ENABLE);
    /* Receive a message */
    ald_uart_recv_by_it(&h_uart, rx_buf, 3);

    while (1)
    {
        ald_rtc_get_time(&time, ALD_RTC_FORMAT_DEC);
        ald_rtc_get_date(&date, ALD_RTC_FORMAT_DEC);
        printf_e("System time: 20%d-%d-%d ", date.year, date.month, date.day);
        printf_e(" %d:%d:%d\r\n", time.hour, time.minute, time.second);

        switch (command)
        {
            /* 01, enter stop1 mode */
            case 1:
                printf_e("MCU enter stop1 mode\r\n");
                ald_pmu_stop1_enter();
                command = 0;
                break;

            /* 02, enter stop2 mode */
            case 2:
                printf_e("MCU enter stop2 mode\r\n");
                ald_pmu_stop2_enter();
                command = 0;
                break;

            /* 03, enter standby mode,  waked by PA1(high) */
            case 3:
                printf_e("MCU enter standby mode, waked by PA1(high)\r\n");
                ald_delay_ms(10);
                ald_pmu_standby_enter(ALD_PMU_PIN_PA7, ALD_PMU_WKPL_HIGH);
                break;

            /* 04, enter standby mode,  waked by PA1(high) and RTC MIN interrupt*/
            case 4:
                ald_rtc_clear_flag_status(ALD_RTC_IF_MIN);
                /* Enable second interrupt */
                ald_rtc_interrupt_config(ALD_RTC_IT_MIN, ENABLE);
                ald_delay_ms(10);

                printf_e("MCU enter standby mode, waked by PA1(high)");
                printf_e(" and RTC MIN interrupt\r\n");
                ald_delay_ms(10);
                ald_pmu_standby_enter(ALD_PMU_PIN_PA7, ALD_PMU_WKPL_HIGH);
                break;

            /* 05, RTC initialization */
            case 5:
                ald_cmu_rtc_clock_select(ALD_CMU_RTC_SEL_LRC);
                /* Initialize RTC */
                init.asynch_pre_div  = 0;
                init.synch_pre_div   = 32767;
                init.hour_format     = ALD_RTC_HOUR_FORMAT_24;
                init.output          = ALD_RTC_OUTPUT_DISABLE;
                init.output_polarity = ALD_RTC_OUTPUT_POLARITY_HIGH;
                ald_rtc_init(&init);

                /* Set current time and date */
                time.hour   = 0;
                time.minute = 0;
                time.second = 0;
                date.day    = 20;
                date.month  = 5;
                date.year   = 24;
                ald_rtc_set_time(&time, ALD_RTC_FORMAT_DEC);
                ald_rtc_set_date(&date, ALD_RTC_FORMAT_DEC);

                command = 0;
                printf_e("RTC initialization\r\n");

                RTC_UNLOCK();
                SET_BIT(RTC->CON, RTC_CON_DSTS_MSK);
                SET_BIT(RTC->CON, RTC_CON_POL_MSK);
                SET_BIT(RTC->CON, RTC_CON_CKOS_MSK);
                RTC_LOCK();

                break;

            /* 06, write data (0x55555555) to backups RAM */
            case 6:
                for (i = 0; i < 32; i++)
                {
                    /* write data to backups RAM (Index: i word) */
                    ald_rtc_bkpc_write_ram(i, BKPC_VALUE_W);
                    printf_e("write data (0x55555555) to backups RAM: %d\r\n", i);
                }

                command = 0;
                break;

            /* 07, read data from backups RAM */
            case 7:
                for (i = 0; i < 32; i++)
                {
                    bkpc_value_read = 0;
                    /* read data from backups RAM (Index: 0 word) */
                    bkpc_value_read = ald_rtc_bkpc_read_ram(i);
                    printf_e("read data from backups RAM: %d-0x%x\r\n", i, bkpc_value_read);
                }

                command = 0;
                break;

            /* 08, read RTC->CON */
            case 8:
                printf_e("read RTC->CON: 0x%x\r\n", RTC->CON);
                printf_e("read RTC->PSR: 0x%x\r\n", RTC->PSR);
                command = 0;
                break;

            /* 09, read PMU->SR & RMU->RSTSR */
            case 9:
                printf_e("read PMU->SR: 0x%x\r\n", PMU->SR);
                printf_e("read RMU->RSTSR: 0x%x\r\n", RMU->RSTSR);

                SYSCFG_UNLOCK();
                RMU->CRSTSR = 0xffffffff;
                SYSCFG_LOCK();

                command = 0;
                break;

            default:
                break;
        }

        ald_delay_ms(1000);
    }
}

/**
  * @brief  GPIO IRQ handler
  * @retval None
  */
__isr__ void gpio_irq_handler(void)
{
    /* Check and clear interrupt flag */
    ald_gpio_exti_get_flag_status(PMU_EXTI_PIN);
    ald_gpio_exti_clear_flag_status(PMU_EXTI_PIN);
    printf_e("gpio_irq_handler\r\n");

    return;
}

/**
  * @brief  check RTC->CON, RTC data&time, backups RAM
  * @retval None.
  */
void system_check(void)
{
    ald_rtc_time_t time;
    ald_rtc_date_t date;
    uint32_t i;

    if (READ_BIT(RTC->CON, RTC_CON_GO_MSK) == 0)
        printf_e("Error: 01, RTC.CON reset\r\n");

    ald_rtc_get_time(&time, ALD_RTC_FORMAT_DEC);
    ald_rtc_get_date(&date, ALD_RTC_FORMAT_DEC);

    /* check time */
    if (time.hour > 23 || time.minute > 59 || time.second > 59 ||
            date.day > 31 || date.month > 12 || date.year > 99)
    {
        printf_e("Error: 02, RTC time overflow\r\n");
    }

    /* check bkrom */
    for (i = 0; i < 32; i++)
    {
        bkpc_value_read = 0;
        /* read data from backups RAM (Index: 0 word) */
        bkpc_value_read = ald_rtc_bkpc_read_ram(i);

        if (bkpc_value_read != BKPC_VALUE_W)
        {
            printf_e("Error: 03, Backups RAM data error\r\n");
            break;
        }
    }

    return;
}

/**
  * @brief  Receive a message complete.
  * @param  arg: Pointer to uart_handle_t structure.
  * @retval None.
  */
void uart_recv_complete(ald_uart_handle_t *arg)
{
    if (rx_buf[0] == 0xFA && rx_buf[2] == 0xFA)
        command = rx_buf[1];

    /* Receive a message */
    ald_uart_recv_by_it(&h_uart, rx_buf, 3);

    return;
}

/**
  * @brief  Output debug information via UART.
  * @param  fmt: Varibale parameter
  * @retval None
  */
void printf_e(const char *fmt, ...)
{
    va_list args;

    va_start(args, fmt);
    vsnprintf(uart_buf, 64, fmt, args);
    va_end(args);

    ald_uart_send(&h_uart, (uint8_t *)uart_buf, strlen(uart_buf), 1000);
    return;
}

/**
  * @brief  Initialize the UART
  * @retval None
  */
void uart_stdio_init(void)
{
    ald_gpio_init_t x;

    /* Initialize tx pin */
    x.mode  = ALD_GPIO_MODE_OUTPUT;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.func  = UART_FUNC_TX;
    ald_gpio_init(UART_PORT_TX, UART_PIN_TX, &x);

    /* Initialize rx pin */
    x.mode  = ALD_GPIO_MODE_INPUT;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.func  = UART_FUNC_RX;
    ald_gpio_init(UART_PORT_RX, UART_PIN_RX, &x);

    /* Initialize uart */
    h_uart.perh             = UART0;
    h_uart.init.baud        = 115200;
    h_uart.init.word_length = ALD_UART_WORD_LENGTH_8B;
    h_uart.init.stop_bits   = ALD_UART_STOP_BITS_1;
    h_uart.init.parity      = ALD_UART_PARITY_NONE;
    h_uart.init.mode        = ALD_UART_MODE_UART;
    h_uart.init.fctl        = ALD_UART_HW_FLOW_CTL_DISABLE;
    h_uart.tx_cplt_cbk      = NULL;
    h_uart.rx_cplt_cbk      = uart_recv_complete;
    h_uart.error_cbk        = NULL;
    ald_uart_init(&h_uart);

    return;
}

/**
  * @brief  Initializate pin of uart module.
  * @retval None
  */
void exti_pin_init(void)
{
    ald_gpio_init_t x;

    /* Initialize PA1 as input */
    x.mode  = ALD_GPIO_MODE_INPUT;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.pupd  = ALD_GPIO_PUSH_DOWN;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.func  = ALD_GPIO_FUNC_1;
    ald_gpio_init(PMU_EXTI_PORT, PMU_EXTI_PIN, &x);
}

/**
  * @}
  */

/**
  * @}
  */
