/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include "main.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
ald_timer_handle_t h_tim;
ald_timer_ic_init_t tim_icinit;
ald_pis_handle_t h_pis;
uint8_t cnt = 0;

void capture_cbk(struct timer_handle_s *arg)
{
    ++cnt;
}

/**
  * @brief  Initializate timer.
  * @retval None.
  */
void timer_init(void)
{
    /* clear timer_handle_t structure */
    memset(&h_tim, 0, sizeof(ald_timer_handle_t));

    /* Initialize GP32C4T1 */
    h_tim.perh           = GP32C4T1;
    h_tim.init.prescaler = 47;
    h_tim.init.mode      = ALD_TIMER_CNT_MODE_UP;
    h_tim.init.period    = 0xFFFF;
    h_tim.init.clk_div   = ALD_TIMER_CLOCK_DIV4;
    h_tim.init.re_cnt    = 0;
    h_tim.capture_cbk    = capture_cbk;
    ald_timer_ic_init(&h_tim);

    tim_icinit.filter   = 0;
    tim_icinit.polarity = ALD_TIMER_IC_POLARITY_RISE;
    tim_icinit.psc      = ALD_TIMER_IC_PSC_DIV1;
    tim_icinit.sel      = ALD_TIMER_IC_SEL_DIRECT;
    ald_timer_ic_config_channel(&h_tim, &tim_icinit, ALD_TIMER_CHANNEL_2);

    /* Start input capture */
    ald_timer_ic_start_by_it(&h_tim, ALD_TIMER_CHANNEL_2);
}

/**
  * @brief  Initializate toggle pin.
  * @retval None.
  */
void pin_init(void)
{
    ald_gpio_init_t x;

    memset(&x, 0, sizeof(ald_gpio_init_t));
    /* Initialize pin */
    x.mode  = ALD_GPIO_MODE_INPUT;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_CMOS;
    x.func  = ALD_GPIO_FUNC_1;
    ald_gpio_init(GPIOB, GPIO_PIN_3, &x);
}

/**
  * @brief  Initializate PIS.
  * @retval None.
  */
void pis_init(void)
{
    /* Clear pis_handle_t struct */
    memset(&h_pis, 0, sizeof(ald_pis_handle_t));
    /* Initialize PIS */
    h_pis.perh                 = PIS;
    h_pis.init.producer_src    = ALD_PIS_GPIOB_PIN3;
    h_pis.init.producer_clk    = ALD_PIS_CLK_PCLK1;
    h_pis.init.producer_edge   = ALD_PIS_EDGE_UP_DOWN;
    h_pis.init.producer_signal = ALD_PIS_OUT_LEVEL;
    h_pis.init.consumer_trig   = ALD_PIS_CH6_TIMER3_CH2IN;
    h_pis.init.consumer_clk    = ALD_PIS_CLK_PCLK2;        //Warning: GP32C4T1 HAND ON PCLK1 ACTUALLY.
    ald_pis_create(&h_pis);
    PIS->TAR_CON0 |= (0x1 << 25);
}


/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    /* Initialize ALD */
    ald_cmu_init();
    ald_cmu_pll_config(ALD_CMU_PLL_INPUT_HRC, ALD_CMU_PLL_OUTPUT_48M);
    ald_cmu_clock_config(ALD_CMU_CLOCK_PLL, 48000000);
    ald_cmu_perh_clock_config(ALD_CMU_PERH_ALL, ENABLE);
    ald_mcu_irq_config(GP32C4T1_IRQn, 1, 1, ENABLE);

    pin_init();
    timer_init();
    pis_init();

    while (1);
}
/**
  * @}
  */

/**
  * @}
  */
