/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include "main.h"
#include <string.h>


/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
ald_timer_handle_t h_tim1;
ald_timer_ic_init_t tim_icinit;
ald_pis_handle_t h_pis;

ald_adc_handle_t h_adc;
ald_adc_nch_conf_t nor_config;
ald_adc_ich_conf_t icf_config;

float nor_value = 0;
float ins_value = 0;
/**
  * @brief  Adc normal convert complete.
  * @param  arg: Pointer to ald_adc_handle_t structure.
  * @retval None.
  */
void normal_convert_complete(ald_adc_handle_t *arg)
{
    return;
}

/**
  * @brief  Adc insert convert complete.
  * @param  arg: Pointer to ald_adc_handle_t structure.
  * @retval None.
  */
void insert_convert_complete(ald_adc_handle_t *arg)
{
    return;
}

/**
  * @brief  Adc level out of window complete.
  * @param  arg: Pointer to ald_adc_handle_t structure.
  * @retval None.
  */
void out_window_complete(ald_adc_handle_t *arg)
{
    return;
}

/**
  * @brief  Adc error complete.
  * @param  arg: Pointer to ald_adc_handle_t structure.
  * @retval None.
  */
void error_complete(ald_adc_handle_t *arg)
{
    return;
}

/**
  * @brief  Adc over complete.
  * @param  arg: Pointer to ald_adc_handle_t structure.
  * @retval None.
  */
void ovr_complete(ald_adc_handle_t *arg)
{
    return;
}

/**
  * @brief  Initializate TTIMER0 BRK pin.
  * @retval None
  */

void timer_pin_init(void)
{
    ald_gpio_init_t x;

    /* Initialize GPIO_PIN_1 as input */
    x.mode  = ALD_GPIO_MODE_INPUT;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.func  = ALD_GPIO_FUNC_2;
    ald_gpio_init(GPIOA, GPIO_PIN_8, &x);
}


/**
  * @brief  Initializate ADC pin.
  * @retval None
  */

void adc_pin_init(void)
{
    ald_gpio_init_t x;

    /* Initialize GPIO_PIN_0 as input */
    x.mode  = ALD_GPIO_MODE_INPUT;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.func  = ALD_GPIO_FUNC_0;
    ald_gpio_init(GPIOC, GPIO_PIN_0, &x);
}

/**
  * @brief  Initializate ADC module.
  * @retval None
  */

void pis_adc_init()
{
    /* clear lpuart_handle_t structure */
    memset(&h_adc, 0x0, sizeof(ald_adc_handle_t));
    /* clear adc_channel_conf_t structure */
    memset(&nor_config, 0x0, sizeof(nor_config));
    /* clear adc_injection_conf_t structure */
    memset(&icf_config, 0x0, sizeof(icf_config));
    /* Enable adc interrupt */
    ald_mcu_irq_config(ADC0_IRQn, 3, 3, ENABLE);
    
    /* Initialize PIS */
    memset(&h_pis, 0, sizeof(ald_pis_handle_t));

    h_pis.perh                 = PIS;
    h_pis.init.producer_src    = ALD_PIS_TIMER0_INPUT_1;
    h_pis.init.producer_clk    = ALD_PIS_CLK_PCLK1;
    h_pis.init.producer_edge   = ALD_PIS_EDGE_NONE;
    h_pis.init.producer_signal = ALD_PIS_OUT_PULSE;
    h_pis.init.consumer_trig   = ALD_PIS_CH6_ADC0_NORMAL;
    h_pis.init.consumer_clk    = ALD_PIS_CLK_PCLK2;
    ald_pis_create(&h_pis);

    /* Initialize adc */
    h_adc.perh            = ADC0;
    h_adc.init.align      = ALD_ADC_DATAALIGN_RIGHT;
    h_adc.init.scan       = DISABLE;
    h_adc.init.cont       = DISABLE;
    h_adc.init.nch_nr     = ALD_ADC_NCH_NR_1;
    h_adc.init.disc       = ALD_ADC_ALL_DISABLE;
    h_adc.init.disc_nr    = ALD_ADC_DISC_NR_1;
    h_adc.init.data_bit   = ALD_ADC_CONV_BIT_12;
    h_adc.init.div        = ALD_ADC_CKDIV_4;
    h_adc.init.nche_sel   = ALD_ADC_NCHESEL_MODE_ONE;
    h_adc.init.n_ref      = ALD_ADC_NEG_REF_VSS;
    h_adc.init.p_ref      = ALD_ADC_POS_REF_VDD;
    h_adc.normal_cplt_cbk = normal_convert_complete;
    h_adc.insert_cplt_cbk = insert_convert_complete;
    h_adc.wdg_cbk         = out_window_complete;
    h_adc.error_cbk       = error_complete;
    h_adc.ovr_cbk         = ovr_complete;
    ald_adc_init(&h_adc);

    /* adc normal external trigger rise edge*/
    ADC_NETS_ENABLE(&h_adc, 0x01);
    ADC_NCH_PIS_CHANNEL_CONFIG(&h_adc, 6);

    /* Initialize regular convert channel */
    nor_config.ch   = ALD_ADC_CHANNEL_0;
    nor_config.idx  = ALD_ADC_NCH_IDX_1;
    nor_config.samp = ALD_ADC_SAMPLETIME_4;
    ald_adc_normal_channel_config(&h_adc, &nor_config);

    ADC_ENABLE(&h_adc);
}

/**
  * @brief  Initializate timer module.
  * @retval None
  */
void pis_tim_init(void)
{
    /* clear timer_handle_t structure */
    memset(&h_tim1, 0, sizeof(ald_timer_handle_t));

    /* Initialize AD16C4T0 */
    h_tim1.perh           = AD16C4T0;
    h_tim1.init.prescaler = 0x00;
    h_tim1.init.mode      = ALD_TIMER_CNT_MODE_UP;
    h_tim1.init.period    = 0xFFFF;
    h_tim1.init.clk_div   = ALD_TIMER_CLOCK_DIV4;
    h_tim1.init.re_cnt    = 0;
    ald_timer_ic_init(&h_tim1);

    tim_icinit.filter   = 3;
    tim_icinit.polarity = ALD_TIMER_IC_POLARITY_RISE;
    tim_icinit.psc      = ALD_TIMER_IC_PSC_DIV1;
    tim_icinit.sel      = ALD_TIMER_IC_SEL_DIRECT;
    ald_timer_ic_config_channel(&h_tim1, &tim_icinit, ALD_TIMER_CHANNEL_1);

    /* Start input capture */
    ald_timer_ic_start(&h_tim1, ALD_TIMER_CHANNEL_1);
}

/**
  * @brief  Start AD16C4T0.
  * @retval None
  */
void pis_tim_start(void)
{
    ald_timer_base_start(&h_tim1);
}

/**
  * @brief  Stop AD16C4T0.
  * @retval None
  */
void pis_tim_stop(void)
{
    ald_timer_base_stop(&h_tim1);
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    /* Initialize ALD */
    ald_cmu_init();
    ald_cmu_pll_config(ALD_CMU_PLL_INPUT_HRC, ALD_CMU_PLL_OUTPUT_48M);
    ald_cmu_clock_config(ALD_CMU_CLOCK_PLL, 48000000);
    ald_cmu_perh_clock_config(ALD_CMU_PERH_ALL, ENABLE);

    /* Initialize timer */
    pis_tim_init();
    timer_pin_init();
    /* Initialize adc/pis */
    adc_pin_init();
    pis_adc_init();
    /* Enable adc interrupt */
    ald_adc_clear_flag_status(&h_adc, ALD_ADC_FLAG_NCH);
    ald_adc_interrupt_config(&h_adc, ALD_ADC_IT_NCH, ENABLE);
    /* start pis trigger adc normal sample*/
    pis_tim_start();

    while (1)
    {
        ald_delay_ms(1000);
    }
}

/**
  * @}
  */

/**
  * @}
  */
