/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include "main.h"
#include <string.h>

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

ald_pis_handle_t h_pis;
ald_timer_handle_t h_tim;
ald_uart_handle_t h_uart;
uint8_t tx_buf[8];
uint8_t rx_buf[8];

/**
  * @brief  Initializate pin of uart module.
  * @retval None
  */
void uart_pin_init(void)
{
    ald_gpio_init_t x;
    /* Initialize tx pin */
    x.mode  = ALD_GPIO_MODE_OUTPUT;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.func  = ALD_GPIO_FUNC_6;
    ald_gpio_init(GPIOC, GPIO_PIN_6, &x); /* PIS ch0 */

    /* Initialize rx pin */
    x.mode  = ALD_GPIO_MODE_INPUT;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.func  = ALD_GPIO_FUNC_1;
    ald_gpio_init(GPIOC, GPIO_PIN_7, &x); 
}

/**
  * @brief  Initializate Uart5 module.
  * @retval None
  */
void pis_uart_init(void)
{
    /* clear ald_timer_handle_t structure */
    memset(&h_uart, 0, sizeof(ald_uart_handle_t));

    h_uart.perh             = UART5;
    h_uart.init.baud        = 115200;
    h_uart.init.word_length = ALD_UART_WORD_LENGTH_8B;
    h_uart.init.stop_bits   = ALD_UART_STOP_BITS_1;
    h_uart.init.parity      = ALD_UART_PARITY_NONE;
    h_uart.init.mode        = ALD_UART_MODE_UART;
    h_uart.init.fctl        = ALD_UART_HW_FLOW_CTL_DISABLE;
    ald_uart_init(&h_uart);
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    /* Initialize ALD */
    ald_cmu_init();
    ald_cmu_pll_config(ALD_CMU_PLL_INPUT_HRC, ALD_CMU_PLL_OUTPUT_48M);
    ald_cmu_clock_config(ALD_CMU_CLOCK_PLL, 48000000);
    ald_cmu_perh_clock_config(ALD_CMU_PERH_ALL, ENABLE);

    /* Clear pis_handle_t struct */
    memset(&h_pis, 0, sizeof(ald_pis_handle_t));
    
    ald_mcu_irq_config(UART5_IRQn, 2, 2, ENABLE);
    
    /* Initialize tx_buf */
    memset(tx_buf, 0x55, sizeof(tx_buf));

    /* Initialize pin */
    uart_pin_init();
    /* Initialize uart5 */
    pis_uart_init();
    /* Initialize PIS RX */
    h_pis.perh                 = PIS;
    h_pis.init.producer_src    = ALD_PIS_GPIOC_PIN7;
    h_pis.init.producer_clk    = ALD_PIS_CLK_PCLK1;
    h_pis.init.producer_edge   = ALD_PIS_EDGE_NONE;
    h_pis.init.producer_signal = ALD_PIS_OUT_LEVEL;
    h_pis.init.consumer_trig   = ALD_PIS_CH14_UART5_RXD;
    h_pis.init.consumer_clk    = ALD_PIS_CLK_PCLK1;
    h_pis.init.input_chan      = ALD_PIS_CHAN_INPUT;
    ald_pis_create(&h_pis);
    
    /* Initialize PIS TX */
    h_pis.perh                 = PIS;
    h_pis.init.producer_src    = ALD_PIS_UART5_TXOUT;
    h_pis.init.producer_clk    = ALD_PIS_CLK_PCLK1;
    h_pis.init.producer_edge   = ALD_PIS_EDGE_NONE;
    h_pis.init.producer_signal = ALD_PIS_OUT_LEVEL;
    h_pis.init.consumer_trig   = ALD_PIS_GPIO_CH0_OUTPUT;   
    h_pis.init.consumer_clk    = ALD_PIS_CLK_PCLK1;
    h_pis.init.input_chan      = ALD_PIS_CHAN_INPUT;
    ald_pis_create(&h_pis);
      
    /* Send message */
    ald_uart_send_by_it(&h_uart, tx_buf, 8);
    ald_delay_ms(1);
    /* Receive message */
    ald_uart_recv_by_it(&h_uart, rx_buf, 8);

    while (1)
    {
        ald_delay_ms(1000);
    }
}
/**
  * @}
  */

/**
  * @}
  */
