/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include <string.h>
#include "main.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

uint8_t tx_buf[8];
uint8_t rx_buf[8];

ald_i2c_handle_t h_i2c;

/**
  * @brief  Initializate pin of i2c module.
  * @retval None
  */
void i2c_pin_init(void)
{
    ald_gpio_init_t x;

    /* Initialize scl pin */
    x.mode  = ALD_GPIO_MODE_OUTPUT;
    x.odos  = ALD_GPIO_OPEN_DRAIN;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.func  = ALD_GPIO_FUNC_5;
    ald_gpio_init(I2C_SCL_PORT, I2C_SCL_PIN, &x);

    /* Initialize sda pin */
    x.mode  = ALD_GPIO_MODE_OUTPUT;
    x.odos  = ALD_GPIO_OPEN_DRAIN;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.func  = ALD_GPIO_FUNC_5;
    ald_gpio_init(I2C_SDA_PORT, I2C_SDA_PIN, &x);

    return;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    /* Initialize ALD */
    ald_cmu_init();
    /* Configure system clock */
    ald_cmu_pll_config(ALD_CMU_PLL_INPUT_HRC, ALD_CMU_PLL_OUTPUT_48M);
    ald_cmu_clock_config(ALD_CMU_CLOCK_PLL, 48000000);
    ald_cmu_perh_clock_config(ALD_CMU_PERH_ALL, ENABLE);

    /* clear i2c_handle_t structure */
    memset(&h_i2c, 0, sizeof(ald_i2c_handle_t));
    /* Initialize i2c pin */
    i2c_pin_init();
    /* Initialize tx_buf */
    memset(tx_buf, 0x55, sizeof(tx_buf));
    /* Enable I2c interrupt */
    ald_mcu_irq_config(I2C1_EV_IRQn, 3, 3, ENABLE);
    ald_mcu_irq_config(I2C1_ERR_IRQn, 3, 3, ENABLE);

    /* Initialize i2c */
    h_i2c.perh = I2C1;
    h_i2c.init.module       = ALD_I2C_MODULE_MASTER;
    h_i2c.init.addr_mode    = ALD_I2C_ADDR_7BIT;
    h_i2c.init.clk_speed    = 100000;
    h_i2c.init.dual_addr    = ALD_I2C_DUALADDR_DISABLE;
    h_i2c.init.general_call = ALD_I2C_GENERALCALL_ENABLE;
    h_i2c.init.no_stretch   = ALD_I2C_NOSTRETCH_ENABLE;
    h_i2c.init.own_addr1    = 0x2D << 1;
    ald_i2c_init(&h_i2c);

    /* Send and Recv data by interrupt */
    ald_i2c_master_send_by_it(&h_i2c, 0x2D << 1, tx_buf, 8);
    ald_delay_ms(1000);
    ald_i2c_master_recv_by_it(&h_i2c, 0x2D << 1, rx_buf, 8);

    while (1)
    {
        ald_delay_ms(1000);
    }
}

/**
  * @}
  */
/**
  * @}
  */
