/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include <string.h>
#include "main.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

uint8_t g_complete = 0;
ald_spi_handle_t h_spi0;
ald_spi_handle_t h_spi1;
uint32_t chip_id = 0;
uint8_t buf_tx[256];
uint8_t buf_rx[256];

/**
  * @brief  Initializate spi tx delay
  * @retval None.
  */
static void __delay(void)
{
    int i = 100;

    while (i--);
}
/**
  * @brief  Read w25q32 id in blocking mode.
  * @retval w25q32 id.
  */
uint32_t meter_bsp_spi_flash_read_id(void)
{
    uint8_t i;
    int r_flag = 0;
    uint8_t flash_id[4] = {0};

    flash_id[0] = 0x90;

    W25Q32_CS_CLR();

    for (i = 0; i < 4; i++)
    {
        if (ald_spi_send_byte_fast(&h_spi0, flash_id[i]) != ALD_OK)
        {
            W25Q32_CS_SET();
            return 0;
        }
    }

    for (i = 0; i < 2; i++)
    {
        flash_id[i] = ald_spi_recv_byte_fast(&h_spi0, &r_flag);

        if (r_flag != ALD_OK)
        {
            W25Q32_CS_SET();
            return 0;
        }
    }

    __delay();
    W25Q32_CS_SET();

    return ((flash_id[0] << 24) | (flash_id[1] << 16) | (flash_id[2] << 8) | (flash_id[3]));
}

/**
  * @brief  Initializate spi pins
  * @retval None.
  */
void spi_pin_init(void)
{
     ald_gpio_init_t x;

    //SPI0
    /* Initialize nss pin */
    x.mode  = ALD_GPIO_MODE_OUTPUT;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.func  = ALD_GPIO_FUNC_1;
    ald_gpio_init(GPIOA, GPIO_PIN_4, &x);
    ald_gpio_write_pin(GPIOA, GPIO_PIN_4, 1);

    /* Initialize sck pin */
    x.mode  = ALD_GPIO_MODE_OUTPUT;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.func  = ALD_GPIO_FUNC_6;
    ald_gpio_init(GPIOA, GPIO_PIN_5, &x);

    /* Initialize miso pin */
    x.mode  = ALD_GPIO_MODE_INPUT;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.func  = ALD_GPIO_FUNC_6;
    ald_gpio_init(GPIOA, GPIO_PIN_6, &x);

    /* Initialize mosi pin */
    x.mode  = ALD_GPIO_MODE_OUTPUT;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.func  = ALD_GPIO_FUNC_6;
    ald_gpio_init(GPIOA, GPIO_PIN_7, &x);
    
    //SPI1
    /* Initialize nss pin */
    x.mode  = ALD_GPIO_MODE_OUTPUT;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.func  = ALD_GPIO_FUNC_1;
    ald_gpio_init(GPIOA, GPIO_PIN_15, &x);
    ald_gpio_write_pin(GPIOA, GPIO_PIN_15, 1);

    /* Initialize sck pin */
    x.mode  = ALD_GPIO_MODE_OUTPUT;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.func  = ALD_GPIO_FUNC_3;
    ald_gpio_init(GPIOB, GPIO_PIN_3, &x);

    /* Initialize miso pin */
    x.mode  = ALD_GPIO_MODE_INPUT;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.func  = ALD_GPIO_FUNC_3;
    ald_gpio_init(GPIOB, GPIO_PIN_4, &x);

    /* Initialize mosi pin */
    x.mode  = ALD_GPIO_MODE_OUTPUT;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.func  = ALD_GPIO_FUNC_3;
    ald_gpio_init(GPIOB, GPIO_PIN_5, &x);
}

/**
  * @brief  Initializate spi0 and spi1
  * @retval None.
  */
void spi_init(void)
{  
    /* Initialize spi */
    h_spi1.perh           = SPI1;
    h_spi1.init.mode      = ALD_SPI_MODE_MASTER;
    h_spi1.init.dir       = ALD_SPI_DIRECTION_2LINES;
    h_spi1.init.data_size = ALD_SPI_DATA_SIZE_8;
    h_spi1.init.baud      = ALD_SPI_BAUD_4;
    h_spi1.init.phase     = ALD_SPI_CPHA_SECOND;
    h_spi1.init.polarity  = ALD_SPI_CPOL_HIGH;
    h_spi1.init.first_bit = ALD_SPI_FIRSTBIT_MSB;
    h_spi1.init.ss_en     = DISABLE;
    h_spi1.init.crc_calc  = DISABLE;
    h_spi1.tx_cplt_cbk    = NULL;
    h_spi1.rx_cplt_cbk    = NULL;
    h_spi1.err_cbk        = NULL;
    ald_spi_init(&h_spi1);
    
    h_spi0.perh           = SPI0;
    h_spi0.init.mode      = ALD_SPI_MODE_MASTER;
    h_spi0.init.dir       = ALD_SPI_DIRECTION_2LINES;
    h_spi0.init.data_size = ALD_SPI_DATA_SIZE_8;
    h_spi0.init.baud      = ALD_SPI_BAUD_4;
    h_spi0.init.phase     = ALD_SPI_CPHA_SECOND;
    h_spi0.init.polarity  = ALD_SPI_CPOL_HIGH;
    h_spi0.init.first_bit = ALD_SPI_FIRSTBIT_MSB;
    h_spi0.init.ss_en     = DISABLE;
    h_spi0.init.crc_calc  = DISABLE;
    h_spi0.tx_cplt_cbk    = NULL;
    h_spi0.rx_cplt_cbk    = NULL;
    h_spi0.err_cbk        = NULL;
    ald_spi_init(&h_spi0);
}

/**
  * @brief  Send the command to read from the flash.
  * @param  addr: Read address.
  * @retval None.
  */
void send_flash_read_cmd(uint32_t addr)
{
    uint8_t i  = 0;
    uint8_t read_cmd = 0x03;
    uint8_t add_buf[5] = {0};

    add_buf[0] = read_cmd;
    add_buf[1] = (addr >> 16) & 0xff;
    add_buf[2] = (addr >> 8) & 0xff;
    add_buf[3] = addr & 0xff;
    
    /* Send flash read cmd */
    for (i = 0; i < 4; i++)
    {
        if (ald_spi_send_byte_fast(&h_spi0, add_buf[i]) != ALD_OK)
        {
            while(1);
        }
    }
}

/**
  * @brief  SPI0 enters receive-only mode, and DMA transfers the received data to be sent by SPI1.
  * @param  size: Data transfer length of DMA.
  * @retval None.
  */
void dma_transmit_tx(uint16_t size)
{
    ald_dma_handle_t hperh;

    /* DMA0 ch0 take received data from SPI0 to SPI1 */
    hperh.perh = DMA0;
    ald_dma_config_struct(&hperh);

    hperh.config.src             = (void *)&SPI0->DATA;
    hperh.config.src_data_width  = ALD_DMA_DATA_SIZE_BYTE;
    hperh.config.src_inc         = ALD_DMA_DATA_INC_DISABLE;
    hperh.config.dst             = (void *)&SPI1->DATA;;
    hperh.config.dst_data_width  = ALD_DMA_DATA_SIZE_BYTE;
    hperh.config.dst_inc         = ALD_DMA_DATA_INC_DISABLE;
    hperh.config.mem_to_mem      = DISABLE;

    hperh.config.size            = size;
    hperh.config.priority        = ALD_DMA_LOW_PRIORITY;
    hperh.config.R_power         = ALD_DMA_R_POWER_1;
    hperh.config.dir             = ALD_DMA_DIR_TO_SRAM;
    hperh.config.circle_mode     = DISABLE;
    hperh.config.msel            = ALD_DMA_MSEL_SPI0;
    hperh.config.msigsel         = ALD_DMA_MSIGSEL_SPI_RNR;
                                 
    hperh.cplt_tc_cbk            = NULL;
    hperh.cplt_tc_arg            = NULL;
    hperh.config.channel         = 0;
    ald_dma_config(&hperh);
    /* Disable DMAMUX single */
    CLEAR_BIT(DMA_MUX->CH_SELCON[0],  DMA_MUX_SINGLE_MSK);    
}

/**
  * @brief Reading data from flash using DMA.
  * @param  addr: The starting address of flash.
  * @param  size: The length of the data to be read.
  * @retval Status of the read operation.
  */
ald_status_t flash_read(uint32_t addr, uint16_t size)
{
    g_complete = 0;
    
    /* Enable SPI0 */
    SPI0->CON1 |= (0X1 << 6);
    
    W25Q32_CS_CLR();
    /* Send flash read cmd */
    send_flash_read_cmd(addr);
    /* Slave CS reset */
    ald_gpio_write_pin(GPIOA, GPIO_PIN_15, 0);
    /* Enable dma0 ch0 */
    DMA0->CH[0].CON |= 0x1; 
    dma_transmit_tx(size);
    /* Enable SPI0 TXDMA/RXDMA */
    SPI0->CON2 |= 0x3U; 
    /* Enable simplex mode of SPI0 to generate SCLK clock */
    ALD_SPI_RXONLY_ENABLE(&h_spi0);
    /* Wait transmit completely */
    while (!g_complete);
    ALD_SPI_RXONLY_DISABLE(&h_spi0);
    /* Slave CS set */
    ald_gpio_write_pin(GPIOA, GPIO_PIN_15, 1);

    W25Q32_CS_SET();

    return ALD_OK;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    /* Initialize ALD */
    ald_cmu_init();
    /* Configure system clock */
    ald_cmu_pll_config(ALD_CMU_PLL_INPUT_HRC, ALD_CMU_PLL_OUTPUT_72M);
    ald_cmu_clock_config(ALD_CMU_CLOCK_PLL, 72000000);
    /* Enable GPIO/UART0 clock */
    ald_cmu_perh_clock_config(ALD_CMU_PERH_ALL, ENABLE);
    
    spi_pin_init();
    spi_init();
    chip_id = meter_bsp_spi_flash_read_id();      
    w25q32_read(0, buf_rx, sizeof(buf_rx));    
    while (1)
    {
        flash_read(0,256);
        ald_delay_ms(10);
    }
}

/**
  * @brief  transmit an amount of data in blocking mode.
  * @param  addr: Specific address which to be write.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be sent
  * @retval Status, see @ref w25_status_t.
  */
w25_status_t w25q32_write(uint32_t addr, uint8_t *buf, uint16_t size)
{
    uint8_t write_cmd  = 0x02;
    uint8_t write_enable_cmd = 0x06;
    uint8_t tx_buf[4] = {0};
    uint16_t i = 0;

    if (buf == NULL)
        return W25_ERROR;

    tx_buf[0] = write_cmd;
    tx_buf[1] = (addr >> 16) & 0xff;
    tx_buf[2] = (addr >> 8) & 0xff;
    tx_buf[3] = addr & 0xff;

    W25Q32_CS_CLR();

    if (ald_spi_send_byte_fast(&h_spi1, write_enable_cmd) != ALD_OK)
    {
        W25Q32_CS_SET();
        return W25_ERROR;
    }

    __delay();
    W25Q32_CS_SET();
    __delay();
    W25Q32_CS_CLR();

    for (i = 0; i < 4; i++)
    {
        if (ald_spi_send_byte_fast(&h_spi1, tx_buf[i]) != ALD_OK)
        {
            W25Q32_CS_SET();
            return W25_ERROR;
        }
    }

    for (i = 0; i < size; i++)
    {
        if (ald_spi_send_byte_fast(&h_spi1, buf[i]) != ALD_OK)
        {
            W25Q32_CS_SET();
            return W25_ERROR;
        }
    }

    __delay();
    W25Q32_CS_SET();
    return W25_OK;
}

/**
  * @brief  Receive an amount of data in blocking mode with fast read mode.
  * @param  addr: address of flash where want to read.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be received
  * @retval Status, see @ref w25_status_t.
  */
w25_status_t w25q32_fast_read(uint32_t addr, uint8_t *buf, uint16_t size)
{
    uint8_t read_cmd = 0x0B;
    uint8_t add_buf[5] = {0};
    uint16_t i = 0;
    int r_flag = 0;

    W25Q32_CS_CLR();

    add_buf[0] = read_cmd;
    add_buf[1] = (addr >> 16) & 0xff;
    add_buf[2] = (addr >> 8) & 0xff;
    add_buf[3] = addr & 0xff;

    if (buf == NULL)
        return W25_ERROR;

    for (i = 0; i < 4; i++)
    {
        if (ald_spi_send_byte_fast(&h_spi1, add_buf[i]) != ALD_OK)
        {
            W25Q32_CS_SET();
            return W25_ERROR;
        }
    }

    ald_spi_recv_byte_fast(&h_spi1, &r_flag);

    for (i = 0; i < size; i++)
    {
        buf[i] = ald_spi_recv_byte_fast(&h_spi1, &r_flag);

        if (r_flag != ALD_OK)
        {
            W25Q32_CS_SET();
            return W25_ERROR;
        }
    }

    W25Q32_CS_SET();

    return W25_OK;
}

/**
  * @brief  Receive an amount of data in blocking mode.
  * @param  addr: address of flash where want to read.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be received
  * @retval Status, see @ref w25_status_t.
  */
w25_status_t w25q32_read(uint32_t addr, uint8_t *buf, uint16_t size)
{
    uint8_t read_cmd = 0x03;
    uint8_t add_buf[5] = {0};
    uint16_t i = 0;
    int r_flag = 0;

    W25Q32_CS_CLR();

    add_buf[0] = read_cmd;
    add_buf[1] = (addr >> 16) & 0xff;
    add_buf[2] = (addr >> 8) & 0xff;
    add_buf[3] = addr & 0xff;

    if (buf == NULL)
        return W25_ERROR;

    for (i = 0; i < 4; i++)
    {
        if (ald_spi_send_byte_fast(&h_spi0, add_buf[i]) != ALD_OK)
        {
            W25Q32_CS_SET();
            return W25_ERROR;
        }
    }

    for (i = 0; i < size; i++)
    {
        buf[i] = ald_spi_recv_byte_fast(&h_spi0, &r_flag);

        if (r_flag != ALD_OK)
        {
            W25Q32_CS_SET();
            return W25_ERROR;
        }
    }

    __delay();
    W25Q32_CS_SET();

    return W25_OK;
}
/**
  * @brief  Erase an sector of data in blocking mode.
  * @param  addr: specific sector address want erase.
  * @retval Status, see @ref w25_status_t.
  */
w25_status_t w25q32_sector_erase(uint8_t addr)
{
    uint8_t sector_erase_cmd = 0x20;
    uint8_t write_enable_cmd = 0x06;
    uint8_t add_buf[4] = {0};
    uint8_t i = 0;

    add_buf[0] = sector_erase_cmd;
    add_buf[1] = (addr >> 16) & 0xff;
    add_buf[2] = (addr >> 8) & 0xff;
    add_buf[3] = addr & 0xff;

    W25Q32_CS_CLR();

    if (ald_spi_send_byte_fast(&h_spi0, write_enable_cmd) != ALD_OK)
    {
        W25Q32_CS_SET();
        return W25_ERROR;
    }

    __delay();
    W25Q32_CS_SET();
    __delay();
    W25Q32_CS_CLR();

    for (i = 0; i < 4; i++)
    {
        if (ald_spi_send_byte_fast(&h_spi0, add_buf[i]) != ALD_OK)
        {
            W25Q32_CS_SET();
            return W25_ERROR;
        }
    }

    __delay();
    W25Q32_CS_SET();

    return W25_OK;
}



/**
  * @}
  */
/**
  * @}
  */
