/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    12 MAR 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 MAR 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include "main.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

ald_uart_handle_t h_uart;
adc_timer_config_t config;
char uart_buf[64];
env_t _env;

ald_pis_handle_t h_pis;
ald_timer_handle_t h_tim;
ald_timer_master_config_t hconfig;
ald_adc_handle_t h_adc;

/**
  * @brief  Output debug information via UART.
  * @param  fmt: Varibale parameter
  * @retval None
  */
void printf_e(const char *fmt, ...)
{
    va_list args;

    va_start(args, fmt);
    vsnprintf(uart_buf, 64, fmt, args);
    va_end(args);

    ald_uart_send(&h_uart, (uint8_t *)uart_buf, strlen(uart_buf), 1000);
    return;
}

/**
  * @brief  Initialize the UART
  * @retval None
  */
void uart_stdio_init(void)
{
    ald_gpio_init_t x;

    /* Initialize tx pin */
    x.mode  = ALD_GPIO_MODE_OUTPUT;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.func  = UART_FUNC_TX;
    ald_gpio_init(UART_PORT_TX, UART_PIN_TX, &x);

    /* Initialize rx pin */
    x.mode  = ALD_GPIO_MODE_INPUT;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.func  = UART_FUNC_RX;
    ald_gpio_init(UART_PORT_RX, UART_PIN_RX, &x);

    /* Initialize uart */
    h_uart.perh             = UART0;
    h_uart.init.baud        = 115200;
    h_uart.init.word_length = ALD_UART_WORD_LENGTH_8B;
    h_uart.init.stop_bits   = ALD_UART_STOP_BITS_1;
    h_uart.init.parity      = ALD_UART_PARITY_NONE;
    h_uart.init.mode        = ALD_UART_MODE_UART;
    h_uart.init.fctl        = ALD_UART_HW_FLOW_CTL_DISABLE;
    h_uart.tx_cplt_cbk      = NULL;
    h_uart.rx_cplt_cbk      = NULL;
    h_uart.error_cbk        = NULL;
    ald_uart_init(&h_uart);

    return;
}

/**
  * @brief  Initializate pin of adc module.
  * @retval None.
  */
void adc_pin_init(void)
{
    ald_gpio_init_t x;

    memset(&x, 0, sizeof(ald_gpio_init_t));

    /* Initialize ADC pin */
    x.mode  = ALD_GPIO_MODE_INPUT;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_CMOS;
    x.func  = ALD_GPIO_FUNC_0;

    /* Initialize ADC pin */
    ald_gpio_init(GPIO_CH0_PORT, ADC_CH0_PIN, &x);
    ald_gpio_init(GPIO_CH2_PORT, ADC_CH2_PIN, &x);
    ald_gpio_init(GPIO_CH3_PORT, ADC_CH3_PIN, &x);
    ald_gpio_init(GPIO_CH9_PORT, ADC_CH9_PIN, &x);

    return;
}

/**
  * @brief  Initializate ADC.
  * @retval None.
  */
void adc_init(void)
{
    ald_adc_nch_conf_t config;

    /* clear structures */
    memset(&h_adc, 0x0, sizeof(ald_adc_handle_t));
    memset(&config, 0x0, sizeof(ald_adc_nch_conf_t));

    /* Initialize adc */
    h_adc.perh            = ADC0;
    h_adc.init.align      = ALD_ADC_DATAALIGN_RIGHT;
    h_adc.init.scan       = ENABLE;
    h_adc.init.cont       = DISABLE;
    h_adc.init.nch_nr     = ALD_ADC_NCH_NR_4;
    h_adc.init.disc       = ALD_ADC_ALL_DISABLE;
    h_adc.init.disc_nr    = ALD_ADC_DISC_NR_1;
    h_adc.init.data_bit   = ALD_ADC_CONV_BIT_12;
    h_adc.init.div        = ALD_ADC_CKDIV_2;
    h_adc.init.nche_sel   = ALD_ADC_NCHESEL_MODE_ONE;
    h_adc.init.n_ref      = ALD_ADC_NEG_REF_VSS;
    h_adc.init.p_ref      = ALD_ADC_POS_REF_VDD;
    h_adc.normal_cplt_cbk = NULL;
    h_adc.insert_cplt_cbk = NULL;
    h_adc.wdg_cbk         = NULL;
    h_adc.error_cbk       = NULL;
    h_adc.ovr_cbk         = NULL;
    ald_adc_init(&h_adc);

    /* Initialize normal convert channel */
    config.ch   = ALD_ADC_CHANNEL_0;
    config.idx  = ALD_ADC_NCH_IDX_1;
    config.samp = ALD_ADC_SAMPLETIME_4;
    ald_adc_normal_channel_config(&h_adc, &config);

    config.ch   = ALD_ADC_CHANNEL_2;
    config.idx  = ALD_ADC_NCH_IDX_2;
    config.samp = ALD_ADC_SAMPLETIME_4;
    ald_adc_normal_channel_config(&h_adc, &config);

    config.ch   = ALD_ADC_CHANNEL_3;
    config.idx  = ALD_ADC_NCH_IDX_3;
    config.samp = ALD_ADC_SAMPLETIME_4;
    ald_adc_normal_channel_config(&h_adc, &config);

    config.ch   = ALD_ADC_CHANNEL_9;
    config.idx  = ALD_ADC_NCH_IDX_4;
    config.samp = ALD_ADC_SAMPLETIME_4;
    ald_adc_normal_channel_config(&h_adc, &config);

    /* adc normal external trigger rise edge*/
    ADC_NETS_ENABLE(&h_adc, ALD_ADC_ETS_RISE);
    ADC_NCH_PIS_CHANNEL_CONFIG(&h_adc, 0x6);

    return;
}

/**
  * @brief  Pis module start.
  * @retval None
  */
void pis_tim_start(void)
{
    ald_timer_base_start(&h_tim);
}

/**
  * @brief  Initializate pis module.
  * @retval None
  */
void pis_tim_init(void)
{
    /* clear ald_timer_handle_t structure */
    memset(&h_tim, 0, sizeof(ald_timer_handle_t));
    memset(&hconfig, 0, sizeof(ald_timer_master_config_t));

    /* Initialize GP16C4T0 */
    h_tim.perh           = GP16C4T0;
    h_tim.init.prescaler = 71;
    h_tim.init.mode      = ALD_TIMER_CNT_MODE_UP;
    h_tim.init.period    = 9999;
    h_tim.init.clk_div   = ALD_TIMER_CLOCK_DIV1;
    h_tim.init.re_cnt    = 0;
    h_tim.period_elapse_cbk = NULL;
    ald_timer_base_init(&h_tim);
    ald_mcu_irq_config(GP16C4T0_IRQn, 0, 3, ENABLE);
    
    /* When updating events of the timer triggered by PIS to other peripherals, 
    the TRGOSEL bit of the CON2 register must be configured, excluding advanced timers */
    hconfig.sel = ALD_TIMER_TRGO_UPDATE;
    ald_timer_master_sync_config(&h_tim, &hconfig);

    /* Enable UPDATE interrupt */
    ald_timer_interrupt_config(&h_tim, ALD_TIMER_IT_UPDATE, ENABLE);

    /* Initialize PIS */
    h_pis.perh                 = PIS;
    h_pis.init.producer_src    = ALD_PIS_TIMER6_TRGOUT;
    h_pis.init.producer_clk    = ALD_PIS_CLK_PCLK1;
    h_pis.init.producer_edge   = ALD_PIS_EDGE_NONE;
    h_pis.init.producer_signal = ALD_PIS_OUT_PULSE;
    h_pis.init.consumer_trig   = ALD_PIS_CH6_ADC0_NORMAL;
    h_pis.init.consumer_clk    = ALD_PIS_CLK_PCLK2;
    ald_pis_create(&h_pis);
}

/**
  * @brief  Initializate toggle pin.
  * @retval None
  */
void toggle_io_init(void)
{
    ald_gpio_init_t x;

    x.mode  = ALD_GPIO_MODE_OUTPUT;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.func  = ALD_GPIO_FUNC_1;
    ald_gpio_init(GPIOB, GPIO_PIN_4, &x); /* Timer toggle pin */
    ald_gpio_init(GPIOB, GPIO_PIN_5, &x); /* DMA toggle pin */
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    /* Initialize ALD */
    ald_cmu_init();
    /* Configure system clock */
    ald_cmu_pll_config(ALD_CMU_PLL_INPUT_HRC, ALD_CMU_PLL_OUTPUT_72M);
    ald_cmu_clock_config(ALD_CMU_CLOCK_PLL, 72000000);
    /* Enable peripheral clock */
    ald_cmu_perh_clock_config(ALD_CMU_PERH_ALL, ENABLE);
    /* Clear env_t structure */
    memset(&_env, 0x0, sizeof(env_t));

    toggle_io_init();

    /* Initialize stdio */
    uart_stdio_init();
    /* Initialize ADC pin */
    adc_pin_init();
    /* Initialize ADC */
    adc_init();

    pis_tim_init();
    ald_adc_start_by_dma(&h_adc, _env.buf, USR_ADC_CH_NR, 0);
    pis_tim_start();

    while (1)
    {
    }
}

/**
  * @}
  */

/**
  * @}
  */
