/**
  *********************************************************************************
  *
  * @file    bsp_key.c
  * @brief   KEY driver
  *
  * @version V1.0
  * @date    24 Apr 2020
  * @author  AE Team
  * @note
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  *********************************************************************************
  */
#include "ald_conf.h"
#include "bsp_key.h"

/** @addtogroup ES32F3xxx_BSP
  * @{
  */

/** @defgroup KEY key
  * @{
  */

/** @defgroup KEY_Private_Params Key Private Variables
  * @{
  */

#define GPIO_PORT_TOP       GPIOA
#define GPIO_PIN_TOP        GPIO_PIN_5

#define GPIO_PORT_BOTTOM    GPIOB
#define GPIO_PIN_BOTTOM     GPIO_PIN_7

#define GPIO_PORT_LEFT      GPIOA
#define GPIO_PIN_LEFT       GPIO_PIN_6

#define GPIO_PORT_RIGHT     GPIOB
#define GPIO_PIN_RIGHT      GPIO_PIN_6

#define GPIO_PORT_CENTER    GPIOD
#define GPIO_PIN_CENTER     GPIO_PIN_2

static key_property_t keys_property[KEY_COUNT];
/* Key fifo variable, struct */
static key_fifo_t keys_fifo;

/**
  * @}
  */

/** @defgroup KEY_Private_Function_Prototypes Key Private Function Prototypes
  * @{
  */
static void bsp_init_key_var(void);
static void bsp_init_key_hard(void);
static void bsp_detect_key(uint8_t i);
/**
  * @}
  */

/** @defgroup KEY_Private_Function Key Private Function
  * @{
  */
/**
  * @brief Read  top key I/O state
  * @retval None
  */
static uint8_t is_key_down_top(void)
{
    if (ald_gpio_read_pin(GPIO_PORT_TOP, GPIO_PIN_TOP) == RESET)
        return 1;
    else
        return 0;
}

/**
  * @brief Read bottom key I/O state.
  * @retval None
  */
static uint8_t is_key_down_bottom(void)
{
    if (ald_gpio_read_pin(GPIO_PORT_BOTTOM, GPIO_PIN_BOTTOM) == RESET)
        return 1;
    else return 0;
}

/**
  * @brief Read left key I/O state.
  * @retval None
  */
static uint8_t is_key_down_left(void)
{
    if (ald_gpio_read_pin(GPIO_PORT_LEFT, GPIO_PIN_LEFT) == RESET)
        return 1;
    else return 0;
}

/**
  * @brief Read right key I/O state.
  * @retval None
  */
static uint8_t is_key_down_right(void)
{
    if (ald_gpio_read_pin(GPIO_PORT_RIGHT, GPIO_PIN_RIGHT) == RESET)
        return 1;
    else return 0;
}

/**
  * @brief Read center key I/O state.
  * @retval None
  */
static uint8_t is_key_down_center(void)
{
    if (ald_gpio_read_pin(GPIO_PORT_CENTER, GPIO_PIN_CENTER) == RESET)
        return 1;
    else return 0;
}

/**
  * @brief Init key hardware.
  * @retval None
  */
static void bsp_init_key_hard(void)
{
    ald_gpio_init_t x;

    x.mode  = ALD_GPIO_MODE_INPUT;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_CMOS;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.nodrv = ALD_GPIO_OUT_DRIVE_0;
    x.podrv = ALD_GPIO_OUT_DRIVE_0;
    x.func  = ALD_GPIO_FUNC_1;

    ald_gpio_init(GPIO_PORT_TOP, GPIO_PIN_TOP, &x);
    ald_gpio_init(GPIO_PORT_BOTTOM, GPIO_PIN_BOTTOM, &x);
    ald_gpio_init(GPIO_PORT_LEFT, GPIO_PIN_LEFT, &x);
    ald_gpio_init(GPIO_PORT_RIGHT, GPIO_PIN_RIGHT, &x);
    ald_gpio_init(GPIO_PORT_CENTER, GPIO_PIN_CENTER, &x);
}

/**
  * @brief Init key variable.
  * @retval None
  */
static void bsp_init_key_var(void)
{
    uint8_t i;

    /* Clear key write/read pointer fifo */
    keys_fifo.read = 0;
    keys_fifo.write = 0;
    keys_fifo.read2 = 0;

    /* Set key struct default value */
    for (i = 0; i < KEY_COUNT; i++)
    {
        keys_property[i].long_time = KEY_LONG_TIME;
        keys_property[i].count = KEY_FILTER_TIME / 2;
        keys_property[i].state = 0;
#if 0
        keys_property[i].KeyCodeDown = 3 * i + 1;
        keys_property[i].KeyCodeUp   = 3 * i + 2;
        keys_property[i].KeyCodeLong = 3 * i + 3;
#endif
        keys_property[i].repeat_speed = 0;
        keys_property[i].repeat_count = 0;
    }

    keys_property[0].key_pressed_detect = is_key_down_top;
    keys_property[1].key_pressed_detect = is_key_down_bottom;
    keys_property[2].key_pressed_detect = is_key_down_left;
    keys_property[3].key_pressed_detect = is_key_down_right;
    keys_property[4].key_pressed_detect = is_key_down_center;
}

/**
  * @brief Detect key state.
  * @param i: key pointer
  * @retval None
  */
static void bsp_detect_key(uint8_t i)
{
    key_property_t *pbtn;

    pbtn = &keys_property[i];

    if (pbtn->key_pressed_detect())
    {
        if (pbtn->count < KEY_FILTER_TIME)
        {
            pbtn->count = KEY_FILTER_TIME;
        }
        else if (pbtn->count < 2 * KEY_FILTER_TIME)
        {
            pbtn->count++;
        }
        else
        {
            if (pbtn->state == 0)
            {
                pbtn->state = 1;
                bsp_put_key((uint8_t)(3 * i + 1));
            }

            if (pbtn->long_time > 0)
            {
                if (pbtn->long_count < pbtn->long_time)
                {
                    if (++pbtn->long_count == pbtn->long_time)
                    {
                        bsp_put_key((uint8_t)(3 * i + 3));
                    }
                }
                else
                {
                    if (pbtn->repeat_speed > 0)
                    {
                        if (++pbtn->repeat_count >= pbtn->repeat_speed)
                        {
                            pbtn->repeat_count = 0;
                            bsp_put_key((uint8_t)(3 * i + 1));
                        }
                    }
                }
            }
        }
    }
    else
    {
        if (pbtn->count > KEY_FILTER_TIME)
        {
            pbtn->count = KEY_FILTER_TIME;
        }
        else if (pbtn->count != 0)
        {
            pbtn->count--;
        }
        else
        {
            if (pbtn->state == 1)
            {
                pbtn->state = 0;
                bsp_put_key((uint8_t)(3 * i + 2));
            }
        }

        pbtn->long_count = 0;
        pbtn->repeat_count = 0;
    }
}

/**
  * @}
  */

/** @defgroup Key_Public_Functions Key Public Functions
  * @{
  */

/**
  * @brief  Init key variable and key hardware
  * @retval None
  */
void bsp_key_init(void)
{
    bsp_init_key_var();
    bsp_init_key_hard();
}

/**
  * @brief  Send a key value in fifo.
  * @param  keycode: key value
  * @retval None
  */
void bsp_put_key(uint8_t keycode)
{
    keys_fifo.buffer[keys_fifo.write] = keycode;

    if (++keys_fifo.write  >= KEY_FIFO_SIZE)
    {
        keys_fifo.write = 0;
    }
}

/**
  * @brief  Get a key value in fifo.
  * @retval Return the first key value
  */
uint8_t bsp_get_key(void)
{
    uint8_t ret;

    if (keys_fifo.read == keys_fifo.write)
    {
        return KEY_NONE;
    }
    else
    {
        ret = keys_fifo.buffer[keys_fifo.read];

        if (++keys_fifo.read >= KEY_FIFO_SIZE)
        {
            keys_fifo.read = 0;
        }

        return ret;
    }
}

/**
  * @brief  Get a key value in fifo.
  * @retval Return the first key value
  */
uint8_t bsp_get_key2(void)
{
    uint8_t ret;

    if (keys_fifo.read2 == keys_fifo.write)
    {
        return KEY_NONE;
    }
    else
    {
        ret = keys_fifo.buffer[keys_fifo.read2];

        if (++keys_fifo.read2 >= KEY_FIFO_SIZE)
        {
            keys_fifo.read2 = 0;
        }

        return ret;
    }
}

/**
  * @brief  Get a key state.
  * @param  keyid: Key id
  * @retval Return key state, 1, pressed; 0, not pressed
  */
uint8_t bsp_get_key_state(key_id_e keyid)
{
    return keys_property[keyid].state;
}

/**
  * @brief  Set key parameter.
  * @param  keyid: Key id
  * @param  longtime: how long time index long-time pressed
  * @param  repeatspeed: repeat speed
  * @retval None
  */
void bsp_set_key_param(uint8_t keyid, uint16_t longtime, uint8_t  repeatspeed)
{
    keys_property[keyid].long_time = longtime;   /* 0, not detect long-time pressed */
    keys_property[keyid].repeat_speed = repeatspeed; /* 0, not support repeat-pressed */
    keys_property[keyid].repeat_count = 0;
}

/**
  * @brief  Clear key fifo.
  * @retval None
  */
void bsp_clear_key(void)
{
    keys_fifo.read = keys_fifo.write;
}

/**
  * @brief Scan key state periodically.
  * @retval None
  */
void bsp_key_scan(void)
{
    uint8_t i;

    for (i = 0; i < KEY_COUNT; i++)
    {
        bsp_detect_key(i);
    }
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

