/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Example DEMO
  *
  * @version V1.0
  * @date    17 Dec 2020
  * @author  AE Team
  * @note
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  *********************************************************************************
  */
#include "main.h"


/* Macro definition of stream buffer */
#define STREAM_BUFFER_TRIGGER_SIZE      1
#define STREAM_BUFFER_MAX_SIZE          20
#define STREAM_BUFFER_TIMEOUT           10000

/* Task functions */
static void vtask_app1(void *pvParameters);
static void vtask_app2(void *pvParameters);
static void vtask_key_scan(void *pvParameters);
static void vtask_stream_buffer(void *pvParameters);
static void app_task_create(void);

/* The variable is stream buffer handle */
static StreamBufferHandle_t xStreamBuffer;

/**
  * @brief  This is main function.
  * @retval Status
  */
int main(void)
{
    ald_cmu_init();
    ald_cmu_pll_config(ALD_CMU_PLL_INPUT_HRC, ALD_CMU_PLL_OUTPUT_72M);
    ald_cmu_clock_config(ALD_CMU_CLOCK_PLL, 72000000);
    ald_cmu_perh_clock_config(ALD_CMU_PERH_ALL, ENABLE);
    ald_systick_interval_select(SYSTICK_INTERVAL_10MS);

    NVIC_SetPriorityGrouping(NVIC_PRIORITY_GROUP_3);

    bsp_led_init();
    bsp_key_init();
    bsp_uart_init();

    app_task_create();

    vTaskStartScheduler();

    while (1);
}

/**
  * @brief  task create function
  * @param  None
  * @retval None
  */
static void app_task_create(void)
{
    xTaskCreate(vtask_app1,
                "vtask_app1",
                512,
                NULL,
                1,
                NULL);

    xTaskCreate(vtask_app2,
                "vtask_app2",
                512,
                NULL,
                2,
                NULL);

    xTaskCreate(vtask_stream_buffer,
                "vtask_stream_buf",
                256,
                NULL,
                3,
                NULL);

    xTaskCreate(vtask_key_scan,       /* task function  */
                "vtask_key_scan",    /* task name */
                256,                 /* size of stack, unit is Word */
                NULL,                /* task parameter */
                4,                   /* task priority */
                NULL);               /* task handler */
}

/**
  * @brief  stream buffer task entry function
  * @param  parameter: user's paramter
  * @retval None
  */
static void vtask_stream_buffer(void *pvParameters)
{
    uint8_t uckey_code;
    uint8_t uccounter = 0, send_size;

    if (NULL == xStreamBuffer)
    {
        while (1)
        {
            printf("Stream buffer pointer is NULL!  \r\n");
            vTaskDelay(pdMS_TO_TICKS(10000));
        }
    }

    while (1)
    {
        uckey_code = bsp_get_key();

        if (uckey_code != KEY_NONE)
        {
            switch (uckey_code)
            {
                case KEY_DOWN_CENTER:
                case KEY_DOWN_TOP:
                case KEY_DOWN_BOTTOM:
                case KEY_DOWN_LEFT:
                case KEY_DOWN_RIGHT:
                    uccounter++;
                    send_size = xStreamBufferSend(xStreamBuffer, &uccounter, sizeof(uccounter), pdMS_TO_TICKS(STREAM_BUFFER_TIMEOUT));

                    if (send_size == sizeof(uccounter))
                    {
                        printf("Stream buffer send %03u.  \r\n", uccounter);
                    }
                    else
                    {
                        printf("Stream buffer send failed.  \r\n");
                    }

                    break;

                default:
                    break;
            }
        }

        vTaskDelay(pdMS_TO_TICKS(20));
    }
}

/**
  * @brief  app1 task entry function
  * @param  parameter: user's paramter
  * @retval None
  */
static void vtask_app1(void *pvParameters)
{
    uint8_t i, receive_size;
    uint8_t rec_buffer[STREAM_BUFFER_MAX_SIZE];

    if (NULL == xStreamBuffer)
    {
        while (1)
        {
            printf("Stream buffer pointer is NULL!  \r\n");
            vTaskDelay(pdMS_TO_TICKS(10000));
        }
    }

    while (1)
    {
        receive_size = xStreamBufferReceive(xStreamBuffer, rec_buffer, sizeof(rec_buffer), pdMS_TO_TICKS(STREAM_BUFFER_TIMEOUT));

        if (receive_size > 0)
        {
            printf("Stream receive data:");

            for (i = 0; i < receive_size; i++)
            {
                printf("  %03d", rec_buffer[i]);
            }

            printf("\r\n");
        }
        else
        {
            printf("Timeout!  \r\n");
        }
    }
}

/**
  * @brief  app2 task entry function
  * @param  parameter: user's paramter
  * @retval None
  */
static void vtask_app2(void *pvParameters)
{
    TickType_t xlast_wake_time;
    const TickType_t xfreq = 400;

    xlast_wake_time = xTaskGetTickCount();

    while (1)
    {
        bsp_led_toggle(1);
        bsp_led_toggle(2);
        vTaskDelayUntil(&xlast_wake_time, pdMS_TO_TICKS(xfreq));
    }
}

/**
  * @brief  key scan task entry function
  * @param  parameter: user's paramter
  * @retval None
  */
static void vtask_key_scan(void *pvParameters)
{
    printf("System start...\r\n");

    xStreamBuffer = xStreamBufferCreate(STREAM_BUFFER_MAX_SIZE, STREAM_BUFFER_TRIGGER_SIZE);

    if (NULL == xStreamBuffer)
    {
        printf("There was not enough heap memory space available to create the stream buffer.  \r\n");
    }
    else
    {
        printf("The stream buffer was created successfully and can now be used.  \r\n");
    }

    while (1)
    {
        bsp_key_scan();
        vTaskDelay(pdMS_TO_TICKS(10));
    }
}
