/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Example DEMO
  *
  * @version V1.0
  * @date    17 Dec 2020
  * @author  AE Team
  * @note
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  *********************************************************************************
  */
#include "main.h"


/* Macro definition of QUEUE*/
#define QUEUE_BUFFER_SIZE       3
#define QUEUE_SEND_TIMEOUT      100
#define QUEUE_RECEIVE_TIMEOUT   10000

/* Task functions */
static void vtask_app1(void *pvParameters);
static void vtask_app2(void *pvParameters);
static void vtask_key_scan(void *pvParameters);
static void vtask_command(void *pvParameters);
static void app_task_create(void);

/* The variable if QUEUE handle*/
QueueHandle_t queue1 = NULL;
QueueHandle_t queue2 = NULL;

/**
  * @brief  This is main function.
  * @retval Status
  */
int main(void)
{
    ald_cmu_init();
    ald_cmu_pll_config(ALD_CMU_PLL_INPUT_HRC, ALD_CMU_PLL_OUTPUT_72M);
    ald_cmu_clock_config(ALD_CMU_CLOCK_PLL, 72000000);
    ald_cmu_perh_clock_config(ALD_CMU_PERH_ALL, ENABLE);
    ald_systick_interval_select(SYSTICK_INTERVAL_10MS);

    NVIC_SetPriorityGrouping(NVIC_PRIORITY_GROUP_3);

    bsp_led_init();
    bsp_key_init();
    bsp_uart_init();

    app_task_create();

    vTaskStartScheduler();

    while (1);
}

/**
  * @brief  task create function
  * @param  None
  * @retval None
  */
static void app_task_create(void)
{
    xTaskCreate(vtask_app1,
                "vtask_app1",
                512,
                NULL,
                1,
                NULL);

    xTaskCreate(vtask_app2,
                "vtask_app2",
                512,
                NULL,
                2,
                NULL);

    xTaskCreate(vtask_command,
                "vtask_command",
                256,
                NULL,
                3,
                NULL);

    xTaskCreate(vtask_key_scan,       /* task function  */
                "vtask_key_scan",    /* task name */
                256,                 /* size of stack, unit is Word */
                NULL,                /* task parameter */
                4,                   /* task priority */
                NULL);               /* task handler */
}

/**
  * @brief  Command task entry function
  * @param  parameter: user's paramter
  * @retval None
  */
static void vtask_command(void *pvParameters)
{
    uint8_t uccount = 0;
    uint8_t uckey_code;

    while (1)
    {
        uckey_code = bsp_get_key();

        if (uckey_code != KEY_NONE)
        {
            switch (uckey_code)
            {
                case KEY_DOWN_LEFT:
                    if (NULL != queue1)
                    {
                        if (pdPASS == xQueueSend(queue1, &uckey_code, pdMS_TO_TICKS(QUEUE_SEND_TIMEOUT)))
                            printf("Queue1 send ok.  \r\n");
                        else
                            printf("Queue1 send failed.  \r\n");
                    }

                    break;

                case KEY_DOWN_RIGHT:
                    if (NULL != queue2)
                    {
                        if (pdPASS == xQueueSend(queue2, &uckey_code, pdMS_TO_TICKS(QUEUE_SEND_TIMEOUT)))
                            printf("Queue2 send ok.  \r\n");
                        else
                            printf("Queue2 send failed.  \r\n");
                    }

                    break;

                default:
                    break;
            }
        }

        vTaskDelay(pdMS_TO_TICKS(10));
    }
}

/**
  * @brief  app1 task entry function
  * @param  parameter: user's paramter
  * @retval None
  */
static void vtask_app1(void *pvParameters)
{
    uint8_t i;
    uint8_t key[QUEUE_BUFFER_SIZE] = {0};

    if (NULL != queue1)
    {
        while (1)
        {
            if (pdTRUE != xQueueReceive(queue1, key, pdMS_TO_TICKS(QUEUE_RECEIVE_TIMEOUT)))
            {
                printf("Queue1 receive timeout.  \r\n");
            }
            else
            {
                printf("Queue1 receive message. message[0]=%d, message[1]=%d, message[2]=%d.  \r\n", key[0], key[1], key[2]);
                bsp_led_toggle(1);
                memset(key, 0, QUEUE_BUFFER_SIZE);
            }
        }
    }
    else
    {
        while (1)
        {
            vTaskDelay(pdMS_TO_TICKS(500));
            bsp_led_toggle(1);
        }
    }
}

/**
  * @brief  app2 task entry function
  * @param  parameter: user's paramter
  * @retval None
  */
static void vtask_app2(void *pvParameters)
{
    uint8_t i;
    uint8_t key[QUEUE_BUFFER_SIZE] = {0};

    if (NULL != queue2)
    {
        while (1)
        {
            if (pdTRUE != xQueueReceive(queue2, key, pdMS_TO_TICKS(QUEUE_RECEIVE_TIMEOUT)))
            {
                printf("Queue2 receive timeout.  \r\n");
            }
            else
            {
                printf("Queue2 receive message. message[0]=%d , message[1]=%d , message[2]=%d ,\r\n", key[0], key[1], key[2]);
                bsp_led_toggle(2);
                memset(key, 0, QUEUE_BUFFER_SIZE);
            }
        }
    }
    else
    {
        while (1)
        {
            vTaskDelay(pdMS_TO_TICKS(500));
            bsp_led_toggle(2);
        }
    }
}

/**
  * @brief  key scan task entry function
  * @param  parameter: user's paramter
  * @retval None
  */
static void vtask_key_scan(void *pvParameters)
{
    printf("System start...\r\n");

    queue1 = xQueueGenericCreate(QUEUE_BUFFER_SIZE, 1, queueQUEUE_TYPE_BASE);

    if (NULL == queue1)
    {
        printf("Creat queue1 failed.  \r\n");
    }
    else
    {
        printf("Creat queue1 ok.  \r\n");
    }

    queue2 = xQueueGenericCreate(QUEUE_BUFFER_SIZE, 1, queueQUEUE_TYPE_BASE);

    if (NULL == queue2)
    {
        printf("Creat queue2 failed.  \r\n");
    }
    else
    {
        printf("Creat queue2 ok.  \r\n");
    }

    while (1)
    {
        bsp_key_scan();
        vTaskDelay(pdMS_TO_TICKS(10));
    }
}
