/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Example DEMO
  *
  * @version V1.0
  * @date    17 Dec 2020
  * @author  AE Team
  * @note
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  *********************************************************************************
  */
#include "main.h"


/* Macro definition of creat software timer numbers*/
#define TIMER_NUMBERS   2

/* Task functions */
static void vtask_app(void *pvParameters);
static void vtask_timer_callback(TimerHandle_t xTimer);

/* Timer functions */
static void timer_create(void);
static void timer_start(void);

/* TimerHandle_t buffer */
static TimerHandle_t timer_handle_buffer[TIMER_NUMBERS] = {NULL};

/**
  * @brief  This is main function.
  * @retval Status
  */
int main(void)
{
    ald_cmu_init();
    ald_cmu_pll_config(ALD_CMU_PLL_INPUT_HRC, ALD_CMU_PLL_OUTPUT_72M);
    ald_cmu_clock_config(ALD_CMU_CLOCK_PLL, 72000000);
    ald_cmu_perh_clock_config(ALD_CMU_PERH_ALL, ENABLE);
    ald_systick_interval_select(SYSTICK_INTERVAL_10MS);

    NVIC_SetPriorityGrouping(NVIC_PRIORITY_GROUP_3);

    bsp_led_init();
    bsp_key_init();
    bsp_uart_init();

    xTaskCreate(vtask_app,            /* task function  */
                "vtask_app",         /* task name */
                256,                 /* size of stack, unit is Word */
                NULL,                /* task parameter */
                4,                   /* task priority */
                NULL);               /* task handler */

    timer_create();

    vTaskStartScheduler();

    while (1);
}

/**
  * @brief  App task entry function
  * @param  parameter: user's paramter
  * @retval None
  */
static void vtask_app(void *pvParameters)
{
    int8_t i;
    TickType_t xlast_wake_time;
    const TickType_t xfreq = 200;

    xlast_wake_time = xTaskGetTickCount();

    timer_start();

    while (1)
    {
        bsp_led_toggle(1);
        vTaskDelayUntil(&xlast_wake_time, pdMS_TO_TICKS(xfreq));
    }
}

/**
  * @brief  Timer create function
  * @param  None
  * @retval None
  */
static void timer_create(void)
{
    uint32_t i;

    for (i = 0; i < TIMER_NUMBERS; i++)
    {
        timer_handle_buffer[i] = xTimerCreate("Timer",  pdMS_TO_TICKS((1000 * i) + 1000), pdTRUE, (void *)i, vtask_timer_callback);

        if (NULL != timer_handle_buffer[i])
        {
            printf("Creat timer%d ok.  \r\n", i);
        }
        else
        {
            printf("Creat timer%d failed.  \r\n", i);
        }
    }
}

/**
  * @brief  Timer start function
  * @param  None
  * @retval None
  */
static void timer_start(void)
{
    int8_t i;

    for (i = 0; i < TIMER_NUMBERS; i++)
    {
        if (NULL == timer_handle_buffer[i])
        {
            printf("Timer%d handle is null.  \r\n", i);
            continue;
        }

        if (pdPASS == xTimerStart(timer_handle_buffer[i], pdMS_TO_TICKS(1000)))
        {
            printf("Timer%d to run.  \r\n", i);
        }
        else
        {
            printf("Timer%d run failed.  \r\n", i);
        }
    }
}

/**
  * @brief  Timer callback function
  * @param  parameter: user's paramter
  * @retval None
  */
static void vtask_timer_callback(TimerHandle_t xTimer)
{
    int32_t timer_id;

    timer_id = (int32_t) pvTimerGetTimerID(xTimer);
    printf("Timer%d is expired.  \r\n", timer_id);
}
