/**
  *********************************************************************************
  *
  * @file    bsp_uart_fifo.c
  * @brief   UART driver
  *
  * @version V1.0
  * @date    16 Apr 2020
  * @author  AE Team
  * @note
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  *********************************************************************************
  */

#include "bsp_uart_fifo.h"
#include "ald_uart.h"
#include "ald_gpio.h"
#include <string.h>
#include "stdio.h"

/** @addtogroup ES32F3xxx_BSP
  * @{
  */

/** @defgroup UART uart
  * @{
  */

/** @defgroup UART_Private_Variables UART Private Variables
  * @{
  */
#if UART0_FIFO_EN == 1
    ald_uart_handle_t h_uart0;
    static uart_transceiver_t isp_uart;
    static uint8_t uart_txbuf0[UART0_TX_BUF_SIZE];
    static uint8_t uart_rxbuf0[UART0_RX_BUF_SIZE];
#endif
#if UART5_FIFO_EN == 1
    uart_handle_t h_uart5;
    static uart_transceiver_t rs485_uart;
    static uint8_t uart_txbuf5[UART5_TX_BUF_SIZE];
    static uint8_t uart_rxbuf5[UART5_RX_BUF_SIZE];
#endif
/**
  * @}
  */

/** @defgroup UART_Static_Function_Prototype UART Static Function Prototype
  * @{
  */
static void uart_soft_init(void);
static void uart_hard_init(void);
static void uart_send(uart_transceiver_t *puart, uint8_t *srcbuf, uint16_t dlen);
static void rs485_ctrl_pin_init(void);
static void rs485_send_before(void);
static void rs485_send_over(void);
static uint8_t uart_get_char_from_fifo(uart_transceiver_t *puart, uint8_t *pbyte);
/**
  * @}
  */

/** @defgroup UART_Static_Function UART Static Function
 * @{
 */
/**
  * @brief  According to idx value to get UART handle pointer
  * @param  idx: ISP_UART or RS485_UART
  * @retval Uart pointer uart handle
  */
uart_transceiver_t *uart_idx_to_handle(uart_idx_e idx)
{
    if (idx == ISP_UART)
    {
#if UART0_FIFO_EN == 1
        return &isp_uart;
#else
        return 0;
#endif
    }
    else if (idx == RS485_UART)
    {
#if UART5_FIFO_EN == 1
        return &rs485_uart;
#else
        return 0;
#endif
    }
    else
    {
        return 0;
    }
}

/**
  * @brief  Uart variable init
  * @retval None
  */
static void uart_soft_init(void)
{
#if UART0_FIFO_EN == 1
    isp_uart.uart = &h_uart0;
    isp_uart.p_txbuf = uart_txbuf0;
    isp_uart.p_rxbuf = uart_rxbuf0;
    isp_uart.txbuf_size = UART0_TX_BUF_SIZE;
    isp_uart.rxbuf_size = UART0_RX_BUF_SIZE;
    isp_uart.idx_txwrite = 0;
    isp_uart.idx_txread = 0;
    isp_uart.idx_rxwrite = 0;
    isp_uart.idx_rxread = 0;
    isp_uart.rx_count = 0;
    isp_uart.tx_count = 0;
    isp_uart.send_before = 0;
    isp_uart.send_over = 0;
    isp_uart.receive = 0;
#endif
#if UART5_FIFO_EN == 1
    rs485_uart.uart   = &h_uart5;
    rs485_uart.p_txbuf = uart_txbuf5;
    rs485_uart.p_rxbuf = uart_rxbuf5;
    rs485_uart.txbuf_size = UART5_TX_BUF_SIZE;
    rs485_uart.rxbuf_size = UART5_RX_BUF_SIZE;
    rs485_uart.idx_txwrite = 0;
    rs485_uart.idx_txread  = 0;
    rs485_uart.idx_rxwrite = 0;
    rs485_uart.idx_rxread  = 0;
    rs485_uart.rx_count = 0;
    rs485_uart.tx_count = 0;
    rs485_uart.send_before = rs485_send_before;
    rs485_uart.send_over  = rs485_send_over;
    rs485_uart.receive = 0;
#endif
}

/**
  * @brief  Rs485 before send to enable the pin
  * @retval None
  */
static void rs485_send_before(void)
{
    RS485_TX_EN();
}

/**
  * @brief  Rs485 send over to disable the pin
  * @retval None
  */
static void rs485_send_over(void)
{
    RS485_RX_EN();
}

/**
  * @brief  Uart hardware init
  * @retval None
  */
static void uart_hard_init(void)
{
    ald_gpio_init_t x;

    /* TX--PB10, RX--PB11 */
#if UART0_FIFO_EN == 1
    x.mode = ALD_GPIO_MODE_OUTPUT;
    x.odos = ALD_GPIO_PUSH_PULL;
    x.pupd = ALD_GPIO_PUSH_UP;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_TTL;
    /* use isp_rx & isp_tx */
    x.func = ALD_GPIO_FUNC_3;
    ald_gpio_init(UART0_TX_PORT, UART0_TX_PIN, &x);
    /* Initialize rx pin ,the same as txpin except mode */
    x.mode = ALD_GPIO_MODE_INPUT;
    ald_gpio_init(UART0_RX_PORT, UART0_RX_PIN, &x);
    /* Initialize uart2 handler */
    h_uart0.perh         = UART0;
    h_uart0.init.baud        = UART0_BAUD;
    h_uart0.init.word_length = ALD_UART_WORD_LENGTH_8B;
    h_uart0.init.stop_bits   = ALD_UART_STOP_BITS_1;
    h_uart0.init.parity      = ALD_UART_PARITY_NONE;
    h_uart0.init.mode        = ALD_UART_MODE_UART;
    h_uart0.init.fctl        = ALD_UART_HW_FLOW_CTL_DISABLE;
    h_uart0.tx_cplt_cbk      = NULL;
    h_uart0.rx_cplt_cbk      = NULL;
    h_uart0.error_cbk        = NULL;
    ald_uart_init(&h_uart0);
    ald_mcu_irq_config(UART0_IRQn, 3, 3, ENABLE);
    SET_BIT(h_uart0.perh->FCON, UART_FCON_RFRST_MSK);
    SET_BIT(h_uart0.perh->ICR, UART_ICR_RFTH_MSK);
    ald_uart_interrupt_config(&h_uart0, ALD_UART_IT_RFTH, ENABLE);
#endif
    /* TX--PB9, RX--PB8 */
#if UART5_FIFO_EN == 1
    x.mode = GPIO_MODE_OUTPUT;
    x.odos = GPIO_PUSH_PULL;
    x.pupd = GPIO_PUSH_UP;
    x.podrv = GPIO_OUT_DRIVE_1;
    x.nodrv = GPIO_OUT_DRIVE_1;
    x.flt  = GPIO_FILTER_DISABLE;
    x.type = GPIO_TYPE_TTL;
    /* use isp_rx & isp_tx */
    x.func = GPIO_FUNC_4;
    ald_gpio_init(UART5_TX_PORT, UART5_TX_PIN, &x);
    /* Initialize rx pin ,the same as txpin except mode */
    x.mode = GPIO_MODE_INPUT;
    ald_gpio_init(UART5_RX_PORT, UART5_RX_PIN, &x);
    /* Initialize uart5 handler */
    h_uart5.perh         = UART5;
    h_uart5.init.baud        = UART5_BAUD;
    h_uart5.init.word_length = UART_WORD_LENGTH_8B;
    h_uart5.init.stop_bits   = UART_STOP_BITS_1;
    h_uart5.init.parity      = UART_PARITY_NONE;
    h_uart5.init.mode        = UART_MODE_UART;
    h_uart5.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
    h_uart5.tx_cplt_cbk      = NULL;
    h_uart5.rx_cplt_cbk      = NULL;
    h_uart5.error_cbk        = NULL;
    ald_uart_init(&h_uart5);
    ald_mcu_irq_config(UART5_IRQn, 3, 3, ENABLE);
    SET_BIT(h_uart5.perh->FCON, UART_FCON_RFRST_MSK);
    SET_BIT(h_uart5.perh->ICR, UART_ICR_RFTH_MSK);
    ald_uart_interrupt_config(&h_uart5, UART_IT_RFTH, ENABLE);
#endif
}

/**
  * @brief  Get a byte from receive buffer
  * @param  puart: pointer to a uart_transceiver_t structure
  * @param  pdes: store data address
  * @retval 0, no data, 1, get data success
  */
static uint8_t uart_get_char_from_fifo(uart_transceiver_t *puart, uint8_t *pdes)
{
    uint16_t cnt;

    DISABLE_INT();
    cnt = puart->rx_count;
    ENABLE_INT();

    if (cnt == 0)
    {
        return 0;
    }
    else
    {
        *pdes = puart->p_rxbuf[puart->idx_rxread];
        DISABLE_INT();

        if (++puart->idx_rxread >= puart->rxbuf_size)
        {
            puart->idx_rxread = 0;
        }

        puart->rx_count--;
        ENABLE_INT();
        return 1;
    }
}

/**
  * @brief  Uart send data
  * @param  puart: pointer to a uart_transceiver_t structure
  * @param  srcbuf: source data address
  * @param  dlen: data length
  * @retval None
  */
static void uart_send(uart_transceiver_t *puart, uint8_t *srcbuf, uint16_t dlen)
{
    uint16_t i;
    __IO uint16_t cnt;

    for (i = 0; i < dlen; i++)
    {
        while (1)
        {
            DISABLE_INT();
            cnt = puart->tx_count;
            ENABLE_INT();

            if (cnt < puart->txbuf_size)
            {
                break;
            }
        }

        puart->p_txbuf[puart->idx_txwrite] = srcbuf[i];
        DISABLE_INT();

        if (++puart->idx_txwrite >= puart->txbuf_size)
        {
            puart->idx_txwrite = 0;
        }

        puart->tx_count++;
        ENABLE_INT();
    }

    SET_BIT(puart->uart->perh->FCON, UART_FCON_TFRST_MSK);
    SET_BIT(puart->uart->perh->ICR, UART_ICR_TFTH_MSK);
    ald_uart_interrupt_config(puart->uart, ALD_UART_IT_TFTH, ENABLE);
}
/**
  * @}
  */

/** @defgroup UART_Public_Function Public Functions
  * @{
  */

/**
  * @brief  Init uart hardware and set gloable variable default value.
  * @retval None
  */
void bsp_uart_init(void)
{
    uart_soft_init();
    uart_hard_init();
    rs485_ctrl_pin_init();
}

/**
  * @brief  Com send data
  * @param  idx: ISP_UART or RS485_UART
  * @param  srcbuf: Source data start address
  * @param  dlen: Data length
  * @retval None
  */
void uart_send_buf(uart_idx_e idx, uint8_t *srcbuf, uint16_t dlen)
{
    uart_transceiver_t *puart;

    puart = uart_idx_to_handle(idx);

    if (puart == 0)
    {
        return;
    }

    if (puart->send_before != 0)
    {
        puart->send_before();
    }

    uart_send(puart, srcbuf, dlen);
}

/**
  * @brief  Com send a byte
  * @param  idx: ISP_UART or RS485_UART
  * @param  dat: send data
  * @retval None
  */
void uart_send_char(uart_idx_e idx, uint8_t dat)
{
    uart_send_buf(idx, &dat, 1);
}

/**
  * @brief  Com get a byte.
  * @param  idx: ISP_UART or RS485_UART
  * @param  pdat: des address
  * @retval 0, no data
  */
uint8_t uart_get_char(uart_idx_e idx, uint8_t *pdat)
{
    uart_transceiver_t *puart;

    puart = uart_idx_to_handle(idx);

    if (puart == 0)
    {
        return 0;
    }

    return uart_get_char_from_fifo(puart, pdat);
}

/**
  * @brief  Com clear tx fifo.
  * @param  idx: ISP_UART or RS485_UART
  * @retval None
  */
void uart_clear_tx_fifo(uart_idx_e idx)
{
    uart_transceiver_t *puart;

    puart = uart_idx_to_handle(idx);

    if (puart == 0)
    {
        return;
    }

    puart->idx_txwrite = 0;
    puart->idx_txread = 0;
    puart->tx_count = 0;
}

/**
  * @brief  Com clear rx fifo.
  * @param  idx: ISP_UART or RS485_UART
  * @retval None
  */
void uart_clear_rx_fifo(uart_idx_e idx)
{
    uart_transceiver_t *puart;

    puart = uart_idx_to_handle(idx);

    if (puart == 0)
    {
        return;
    }

    puart->idx_rxwrite = 0;
    puart->idx_rxread = 0;
    puart->rx_count = 0;
}

/**
  * @brief  Rs485 control pin init
  * @retval None
  */
void rs485_ctrl_pin_init(void)
{
    ald_gpio_init_t x;

    x.mode  = ALD_GPIO_MODE_OUTPUT;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_CMOS;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.func  = ALD_GPIO_FUNC_1;
    ald_gpio_init(PORT_RS485_TXEN, PIN_RS485_TXEN, &x);
    ald_gpio_write_pin(PORT_RS485_TXEN, PIN_RS485_TXEN, 0);
}

/**
  * @brief  Rs485 send data.
  * @param  srcbuf: send data source data start address
  * @param  dlen: data length
  * @retval None
  */
void rs485_send_buf(uint8_t *srcbuf, uint16_t dlen)
{
    uart_send_buf(RS485_UART, srcbuf, dlen);
}

/**
  * @brief  Rs485 send string.
  * @param  pbuf: string source address
  * @retval None
  */
void rs485_send_str(char *pbuf)
{
    rs485_send_buf((uint8_t *)pbuf, strlen(pbuf));
}

/**
  * @brief  Uart interrupt handler
  * @param  puart: pointer to a uart_transceiver_t structure
  * @retval None
  */
static void uart_irq(uart_transceiver_t *puart)
{
    uint8_t ch;

    /* Receive handler */
    if ((ald_uart_get_mask_flag_status(puart->uart, ALD_UART_IF_RFTH)) != RESET)
    {
        ald_uart_clear_flag_status(puart->uart, ALD_UART_IF_RFTH);
        ch = (uint8_t)(puart->uart->perh->RXBUF & 0xFF);
        puart->p_rxbuf[puart->idx_rxwrite] = ch;

        if (++puart->idx_rxwrite >= puart->rxbuf_size)
        {
            puart->idx_rxwrite = 0;
        }

        if (puart->rx_count < puart->rxbuf_size)
        {
            puart->rx_count++;
        }
    }

    /* Transmit handler */
    if ((ald_uart_get_mask_flag_status(puart->uart, ALD_UART_IF_TFTH) != RESET))
    {
        ald_uart_clear_flag_status(puart->uart, ALD_UART_IF_TFTH);

        if (puart->tx_count == 0)
        {
            ald_uart_interrupt_config(puart->uart, ALD_UART_IT_TFTH, DISABLE);
            ald_uart_interrupt_config(puart->uart, ALD_UART_IT_TBC, ENABLE);
        }
        else
        {
            puart->uart->perh->TXBUF = puart->p_txbuf[puart->idx_txread];

            if (++puart->idx_txread >= puart->txbuf_size)
            {
                puart->idx_txread = 0;
            }

            puart->tx_count--;
        }
    }
    else if ((ald_uart_get_mask_flag_status(puart->uart, ALD_UART_IF_TBC) != RESET))
    {
        if (puart->tx_count == 0)
        {
            while (puart->uart->perh->STAT & ALD_UART_STATUS_TSBUSY);

            ald_uart_interrupt_config(puart->uart, ALD_UART_IT_TBC, DISABLE);
            ald_uart_clear_flag_status(puart->uart, ALD_UART_IF_TBC);

            if (puart->send_over)
            {
                puart->send_over();
            }
        }
    }
}

/**
  * @brief  UART2 interrupt handler
  * @retval None
  */
#if UART0_FIFO_EN == 1
void UART0_Handler(void)
{
    uart_irq(&isp_uart);
}
#endif

/**
  * @brief  UART5 interrupt handler
  * @retval None
  */
#if UART5_FIFO_EN == 1
void UART5_Handler(void)
{
    uart_irq(&rs485_uart);
}
#endif

/**
  * @brief  UART2 receive frame monitor function
  * @param  idx: ISP_UART or RS485_UART
  * @retval None
  */
void uart0_frame_monitor(uart_idx_e idx)
{
    static uint16_t cntbkp = 0;
    static uint8_t  idletmr = 0;
    uint8_t temp = 0;
    uart_transceiver_t *puart;

    puart = uart_idx_to_handle(idx);

    if (puart == 0)
    {
        return;
    }

    temp = puart->rx_count;

    if (temp > 0)
    {
        if (cntbkp != temp)
        {
            cntbkp = temp;
            idletmr = 0;
        }
        else
        {
            if (idletmr < 30)
            {
                idletmr++;

                if (idletmr >= 30)
                {
                    puart->frame_flag = 1;
                }
            }
        }
    }
    else
    {
        cntbkp = 0;
    }
}

/**
  * @brief  UART5 receive frame monitor function
  * @param  idx: ISP_UART or RS485_UART
  * @retval None
  */
void uart5_frame_monitor(uart_idx_e idx)
{
    static uint16_t cntbkp = 0;
    static uint8_t  idletmr = 0;
    uint8_t temp = 0;
    uart_transceiver_t *puart;

    puart = uart_idx_to_handle(idx);

    if (puart == 0)
    {
        return;
    }

    temp = puart->rx_count;

    if (temp > 0)
    {
        if (cntbkp != temp)
        {
            cntbkp = temp;
            idletmr = 0;
        }
        else
        {
            if (idletmr < 3)
            {
                idletmr++;

                if (idletmr >= 3)
                {
                    puart->frame_flag = 1;
                }
            }
        }
    }
    else
    {
        cntbkp = 0;
    }
}

/**
  * @brief  Get frame flag
  * @param  idx: ISP_UART or RS485_UART
  * @retval Frame flag
  */
uint8_t uart_get_frame_flag(uart_idx_e idx)
{
    uart_transceiver_t *puart;

    puart = uart_idx_to_handle(idx);

    if (puart == 0)
    {
        return 0;
    }

    return  puart->frame_flag;
}

/**
  * @brief  Clear frame flag
  * @param  idx: ISP_UART or RS485_UART
  * @retval None
  */
void uart_clear_frame_flag(uart_idx_e idx)
{
    uart_transceiver_t *puart;

    puart = uart_idx_to_handle(idx);
    __disable_irq();
    puart->frame_flag = 0;
    __enable_irq();
}

/**
  * @brief  Get the received frame
  * @param  idx: ISP_UART or RS485_UART
  * @retval None
  */
uint8_t uart_get_frame(uart_idx_e idx, uint8_t *desbuf)
{
    uint8_t length = 0;
    uint8_t i = 0;
    uart_transceiver_t *puart;

    puart = uart_idx_to_handle(idx);

    if (puart == 0)
    {
        return 0;
    }

    __disable_irq();
    length = puart->rx_count;
    __enable_irq();

    if (length == 0)return 0;

    for (i = 0; i < length; i++)
    {
        desbuf[i] = puart->p_rxbuf[puart->idx_rxread];
        __disable_irq();
        puart->idx_rxread++;

        if (puart->idx_rxread >= puart->rxbuf_size)
        {
            puart->idx_rxread = 0;
        }

        puart->rx_count--;
        __enable_irq();
    }

    return length;
}

/**
  * @}
  */

/** @defgroup UART_Printf_Remap_Function UART Printf Remap Functions
  * @{
  */
/**
  * @brief  Reallocate the printf fputc
  * @param  ch: promotion of the character to be written
  * @param  f : pointer to a FILE object
  * @retval Write state
  */
int fputc(int ch, FILE *f)
{
    h_uart0.perh->TXBUF = (uint8_t)ch;

    while ((ald_uart_get_status(&h_uart0, ALD_UART_STATUS_TFTH)) != SET);

    return ch;
}

/**
  * @brief  Reallocate the fgetc
  * @param  f : pointer to a FILE object
  * @retval Write state
  */
int fgetc(FILE *f)
{
#if 0

    while (uart_get_char(ISP_UART, &data) == 0);

#endif
    return 1;
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */


