/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Example DEMO
  *
  * @version V1.0
  * @date    17 Dec 2020
  * @author  AE Team
  * @note
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  *********************************************************************************
  */
#include "main.h"


/* Task functions */
static void vtask_app1(void *pvParameters);
static void vtask_app2(void *pvParameters);
static void vtask_key_scan(void *pvParameters);
static void vtask_info_print(void *pvParameters);
static void app_task_create(void);

/**
  * @brief  This is main function.
  * @retval Status
  */
int main(void)
{
    ald_cmu_init();
    ald_cmu_pll_config(ALD_CMU_PLL_INPUT_HRC, ALD_CMU_PLL_OUTPUT_72M);
    ald_cmu_clock_config(ALD_CMU_CLOCK_PLL, 72000000);
    ald_cmu_perh_clock_config(ALD_CMU_PERH_ALL, ENABLE);
    ald_systick_interval_select(SYSTICK_INTERVAL_10MS);

    NVIC_SetPriorityGrouping(NVIC_PRIORITY_GROUP_3);

    bsp_led_init();
    bsp_key_init();
    bsp_uart_init();

    app_task_create();

    vTaskStartScheduler();

    while (1);
}

/**
  * @brief  task create function
  * @param  None
  * @retval None
  */
static void app_task_create(void)
{
    xTaskCreate(vtask_app1,
                "vtask_app1",
                512,
                NULL,
                1,
                NULL);

    xTaskCreate(vtask_app2,
                "vtask_app2",
                512,
                NULL,
                2,
                NULL);

    xTaskCreate(vtask_info_print,
                "vtask_info_print",
                256,
                NULL,
                3,
                NULL);

    xTaskCreate(vtask_key_scan,          /* task function  */
                "vtask_key_scan",       /* task name */
                256,                    /* size of stack, unit is Word */
                NULL,                   /* task parameter */
                4,                      /* task priority */
                NULL);                  /* task handler */
}

/**
  * @brief  information print task entry function
  * @param  parameter: user's paramter
  * @retval None
  */
static void vtask_info_print(void *pvParameters)
{
    uint8_t uccount = 0;
    uint8_t uckey_code;

    while (1)
    {
        uckey_code = bsp_get_key();

        if (uckey_code != KEY_NONE)
        {
            switch (uckey_code)
            {
                case KEY_DOWN_CENTER:
                    printf("Center key pressed down. \r\n");
                    break;

                case KEY_DOWN_TOP:
                    printf("TOP key pressed down. \r\n");
                    break;

                case KEY_DOWN_BOTTOM:
                    printf("BOTTOM key pressed down. \r\n");
                    break;

                case KEY_DOWN_LEFT:
                    printf("LEFT key pressed down. \r\n");
                    break;

                case KEY_DOWN_RIGHT:
                    printf("RIGHT key pressed down. \r\n");
                    break;

                default:
                    break;
            }
        }

        vTaskDelay(pdMS_TO_TICKS(10));
    }
}

/**
  * @brief  app1 task entry function
  * @param  parameter: user's paramter
  * @retval None
  */
static void vtask_app1(void *pvParameters)
{
    TickType_t xlast_wake_time;
    const TickType_t xfreq = 200;

    xlast_wake_time = xTaskGetTickCount();

    while (1)
    {
        bsp_led_toggle(1);
        vTaskDelayUntil(&xlast_wake_time, pdMS_TO_TICKS(xfreq));
    }
}

/**
  * @brief  app2 task entry function
  * @param  parameter: user's paramter
  * @retval None
  */
static void vtask_app2(void *pvParameters)
{
    TickType_t xlast_wake_time;
    const TickType_t xfreq = 200;

    xlast_wake_time = xTaskGetTickCount();

    while (1)
    {
        bsp_led_toggle(2);
        vTaskDelayUntil(&xlast_wake_time, pdMS_TO_TICKS(xfreq));
    }
}

/**
  * @brief  key scan task entry function
  * @param  parameter: user's paramter
  * @retval None
  */
static void vtask_key_scan(void *pvParameters)
{
    printf("System start...\r\n");

    while (1)
    {
        bsp_key_scan();
        vTaskDelay(pdMS_TO_TICKS(10));
    }
}
