/**
  *********************************************************************************
  *
  * @file    bsp_uart_fifo.h
  * @brief   uart driver
  *
  * @version V1.0
  * @date    16 Apr 2020
  * @author  AE Team
  * @note
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  *********************************************************************************
  */
#ifndef _BSP_USART_FIFO_H_
#define _BSP_USART_FIFO_H_

#include "ald_uart.h"

/** @addtogroup ES32F03xxx_BSP
  * @{
  */

/** @addtogroup UART
  * @{
  */

/** @defgroup UART_Private_Micros UART Private Micros
  * @{
  */

#define UART0_FIFO_EN   1

#define UART5_FIFO_EN   0
/* UART0 -ISP */
#define UART0_TX_PORT    GPIOB
#define UART0_TX_PIN     GPIO_PIN_10
#define UART0_RX_PORT    GPIOB
#define UART0_RX_PIN     GPIO_PIN_11
/* UART5 -RS485 */
#define UART5_TX_PORT    GPIOB
#define UART5_TX_PIN     GPIO_PIN_9
#define UART5_RX_PORT    GPIOB
#define UART5_RX_PIN     GPIO_PIN_8
#define PORT_RS485_TXEN  GPIOC
#define PIN_RS485_TXEN   GPIO_PIN_13
#define RS485_RX_EN()    ald_gpio_write_pin(GPIOC, GPIO_PIN_13, 0)
#define RS485_TX_EN()    ald_gpio_write_pin(GPIOC, GPIO_PIN_13, 1)
#define ENABLE_INT()    __set_PRIMASK(0)
#define DISABLE_INT()   __set_PRIMASK(1)

/* UART index */
typedef enum
{
    ISP_UART    = 0,    /* UART0, PB10, PB11 */
    RS485_UART  = 5,    /* UART5, PB9, PB8 */
} uart_idx_e;

#if UART0_FIFO_EN == 1
    #define UART0_BAUD      115200
    #define UART0_TX_BUF_SIZE   1*512
    #define UART0_RX_BUF_SIZE   1*512
#endif
#if UART5_FIFO_EN == 1
    #define UART5_BAUD      9600
    #define UART5_TX_BUF_SIZE   1*512
    #define UART5_RX_BUF_SIZE   1*512
#endif

typedef struct
{
    ald_uart_handle_t *uart;
    uint8_t *p_txbuf;
    uint8_t *p_rxbuf;
    uint16_t txbuf_size;
    uint16_t rxbuf_size;
    __IO uint16_t idx_txwrite;
    __IO uint16_t idx_txread;
    __IO uint16_t tx_count;
    __IO uint8_t frame_flag;
    __IO uint16_t idx_rxwrite;
    __IO uint16_t idx_rxread;
    __IO uint16_t rx_count;
    void (*send_before)(void);
    void (*send_over)(void);
    void (*receive)(uint8_t _byte);
} uart_transceiver_t;
/**
  * @}
  */

/** @addtogroup UART_Public_Functions
  * @{
  */
void bsp_uart_init(void);
void uart_send_buf(uart_idx_e idx, uint8_t *srcbuf, uint16_t dlen);
void uart_send_char(uart_idx_e idx, uint8_t dat);
uint8_t uart_get_char(uart_idx_e idx, uint8_t *des);
void uart_clear_tx_fifo(uart_idx_e idx);
void uart_clear_rx_fifo(uart_idx_e idx);
void uart5_frame_monitor(uart_idx_e idx);
void uart0_frame_monitor(uart_idx_e port);
uint8_t uart_get_frame_flag(uart_idx_e idx);
void uart_clear_frame_flag(uart_idx_e idx);
uint8_t uart_get_frame(uart_idx_e idx, uint8_t *desbuf);
void rs485_send_buf(uint8_t *srcbuf, uint16_t dlen);
void rs485_send_str(char *srcbuf);
/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
#endif



