/**
  *********************************************************************************
  * @file   boot_flash.c
  * @brief  Write update to flash or memory.
  *
  * @version V1.0
  * @date    31 Dec 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          31 Dec 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */
#include <string.h>
#include <stdio.h>
#include "ald_crc.h"
#include "ald_syscfg.h"
#include "boot_flash.h"
#include "boot_frame.h"
#include "boot_shell.h"

/** @addtogroup Bootloader
  * @{
  */

/** @defgroup Flash Flash
  * @brief    Bootloader Flash Module
  * @{
  */

/** @defgroup Flash_Public_Functions Public Functions
  * @brief    Flash Public Functions
  * @{
  */
/**
  * @brief  Get page of data flash
  * @retval Number of page
  */
uint32_t flash_usr_page_get(void)
{
    uint32_t *p = NULL;
    uint32_t tmp = 0;

    p = (uint32_t *)APP_FLASH_S;

    while (*p++ != 0xFFFFFFFF)
    {
        tmp++;
    }
    
    return ((tmp + 128) / 256);
}

/**
  * @brief  Erase all usr flash
  * @retval 0-success, other value indicates failed.
  */
int flash_usr_page_erase(void)
{
    uint32_t addr = APP_FLASH_S;
    
    while ((addr + APP_FLASH_PAGE_SIZE) <= APP_FLASH_E+1)
    {
        if (ald_iap_erase_page(addr))
            return -1;

        addr += APP_FLASH_PAGE_SIZE;
    }

    return 0;
}

/**
  * @brief  write data to flash at specified address.
  * @param  page: specific page.
  * @param  data: write data
  * @param  crc: data crc.
  * @retval 0-success, other value indicates failed.
  */
int write_data_to_flash(uint8_t page, uint8_t *data, uint16_t crc)
{
    uint8_t *p;
    uint16_t _crc;

    p = (uint8_t *)USR_FLASH_ADDER(page);

    if (ald_iap_program_words(USR_FLASH_ADDER(page), data, 1024, ENABLE))
        return -1;

    __disable_irq();

    SYSCFG_UNLOCK();
    SYSCFG_BOOTFLASH_MAPPING_DISABLE();
    SYSCFG_LOCK();

    CRC_RESET(&h_crc);
    _crc = ald_crc_calculate(&h_crc, p, 1024);

    SYSCFG_UNLOCK();
    SYSCFG_BOOTFLASH_MAPPING_ENABLE();
    SYSCFG_LOCK();

    __enable_irq();

    if (_crc != crc)
        return -2;

    return 0;
}
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
