/**
  *********************************************************************************
  * @file   boot.c
  * @brief  Base functions.
  *
  * @version V1.0
  * @date    31 Dec 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          31 Dec 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */
#include <string.h>
#include <stdio.h>
#include "ald_syscfg.h"
#include "ald_rmu.h"
#include "boot.h"
#include "boot_flash.h"
#include "boot_frame.h"
#include "boot_shell.h"


/** @defgroup Bootloader Bootloader
  * @brief    Eastsoft Bootloader
  * @{
  */

/** @defgroup Base Base
  * @brief    Bootloader Base
  * @{
  */
/** @defgroup Base_Public_Variables Public Variable
  * @brief    Base Public Variable
  * @{
  */
timer_env_t timer_env = {0};
extern ald_crc_handle_t h_crc;
/**
  * @}
  */

/** @defgroup Base_Private_Functions Private Functions
  * @brief    Base Private_Functions
  * @{
  */
/**
  * @brief  Systick callback function
  * @retval None
  */
__isr__ void ald_systick_irq_cbk(void)
{
    if (!timer_env.state)
        return;

    if ((--timer_env.tick == 0) && (timer_env.cbk != NULL))
        timer_env.cbk();
}

/**
  * @brief  Initialize timer.
  * @retval None
  */
void timer_init(void)
{
    memset(&timer_env, 0x0, sizeof(timer_env_t));
    return;
}

/**
  * @brief  Start timer.
  * @param  tick: Timeout in milliseconds.
  * @param  cbk: callback function for timeout.
  * @retval None
  */
void timer_start(uint32_t tick, timer_cbk cbk)
{
    tick = tick == 0 ? 1 : tick;

    __disable_irq();
    timer_env.tick  = tick;
    timer_env.cbk   = cbk;
    timer_env.state = 1;
    __enable_irq();
}

/**
  * @brief  Stop timer.
  * @retval None
  */
void timer_stop(void)
{
    __disable_irq();
    memset(&timer_env, 0x0, sizeof(timer_env_t));
    __enable_irq();
}

/**
  * @brief  Callback function for timeout.
  * @retval None
  */
void boot_timer_cbk(void)
{
    timer_env.timeout = 1;
}

#if 1
/**
  * @brief  Check app CRC
  * @param  page: total page of flash data.
  * @retval 0-success, other value indicates failed.
  */
uint8_t check_app_crc(void)
{

    uint8_t result = 0;
    uint32_t begin, end;
    uint32_t applen;
    uint32_t appcrc;
    
    appcrc = *((uint32_t *)(APP_FLASH_INFO_S + 0));
    applen = *((uint32_t *)(APP_FLASH_INFO_S + 4));
    begin = ADDR_CHECK_APP_FLASH_CRC_START;
    end = ADDR_CHECK_APP_FLASH_CRC_START + applen;

    if (!VALID_APP_ADDR(end))
    {
        return 1;
    }

    if ((begin & 0x3) || (end & 0x3) || (end > (APP_FLASH_S + APP_FLASH_SIZE)))
        return 2;

    /* Initialize CRC */
    h_crc.perh          = CRC;
    h_crc.init.mode     = ALD_CRC_MODE_32;
    h_crc.init.seed     = 0xFFFFFFFF;
    h_crc.init.data_rev = ENABLE;
    h_crc.init.data_inv = DISABLE;
    h_crc.init.chs_rev  = ENABLE;
    h_crc.init.chs_inv  = ENABLE;
    ald_crc_init(&h_crc);

    if (ald_crc_calculate(&h_crc, (uint8_t *)begin, applen) != appcrc)
        result = 1;
    else
        result = 0;
        
    return result;
}

#endif

/**
  * @brief  Copy data from flash to sram.
  * @param  page: total page of flash data.
  * @retval 0-success, other value indicates failed.
  */
int copy_data_to_sram(uint8_t page)
{
    uint32_t *p_flash = NULL;
    uint32_t *p_ram   = NULL;
    uint32_t tmp = 0;

    if (page > ((APP_RUN_RAM_E - APP_RUN_RAM_S) / 1024))
    {
        printf_e("\r\nCopy data to ram error, Exceeded RAM size!!!");
        return -1;
    }

    p_flash = (uint32_t *)APP_FLASH_S;
    p_ram   = (uint32_t *)(APP_RUN_RAM_S);

    for (tmp = 0; tmp < (page * 256); tmp++)
    {
        *p_ram++ = *p_flash++;
    }

    return 0;
}

/**
  * @brief  Run in flash
  * @retval None.
  */
void run_in_flash(void)
{
    uint32_t addr = 0;

    SYSCFG_UNLOCK();
    SYSCFG_BOOTFLASH_MAPPING_DISABLE();
    SYSCFG_LOCK();

    ald_uart_reset(frame_env.h_uart);
    ald_crc_reset(&h_crc);

    addr = *(uint32_t *)(APP_FLASH_S + 4);
    ald_vtor_config(OFFSET_VTOR_FLASH, ENABLE);
    __set_MSP(*(volatile uint32_t *)APP_FLASH_S);
    ((void(*)(void))(addr))();
}

/**
  * @brief  Run in ram
  * @retval None.
  */
void run_in_sram(void)
{
    uint32_t addr = 0;

    ald_uart_reset(frame_env.h_uart);
    ald_crc_reset(&h_crc);

    addr = *(uint32_t *)(APP_RUN_RAM_S + 4);
    ald_vtor_config(OFFSET_VTOR_RAM, ENABLE);
    __set_MSP(*(volatile uint32_t *)APP_RUN_RAM_S);
    ((void(*)(void))(addr))();
}

/**
  * @brief  command for "version".
  * @retval None
  */
void cmd_version(void)
{
    printf_e("\r\nBootloader: 01-00-00-00");
    printf_e("\r\nTime: " __DATE__ " " __TIME__);
    return;
}

/**
  * @brief  command for "reboot".
  * @retval None
  */
void cmd_reboot(void)
{
    ald_rmu_reset_periperal(ALD_RMU_PERH_CHIP);
}

static void uart0_irq_ctrl(type_func_t status)
{
    ald_mcu_irq_config(UART0_IRQn, 1, 3, status);
}

/**
  * @brief  command for "download_ram".
  * @retval None
  */
void cmd_download(void)
{
    uart0_irq_ctrl(DISABLE);
    printf_e("\r\n************* Update Program *************");
    printf_e("\r\nStrat updating afer 3 second !");
    printf_e("\r\nDon't input anything!!!");
    ald_delay_ms(3000);

    flash_usr_page_erase();
    boot_image_update();
    ald_delay_ms(100);
    boot_update_info();
    /* Clear UART0 interrupt flag */
    UART0->ICR = 0XFFFFFFFF;
    uart0_irq_ctrl(ENABLE);
}

/**
  * @brief  command for "run_flash".
  * @retval None
  */
void run_flash_cmd(void)
{
    if (!flash_usr_page_get())
    {
        printf_e("The APP flash is empty, Please Update first!\r\n");
        return;
    }

    printf_e("\r\n");
    run_in_flash();
    return;
}

/**
  * @brief  command for "run_ram".
  * @retval None
  */
void run_sram_cmd(void)
{
    uint8_t page = flash_usr_page_get();

    if (copy_data_to_sram(page))
        return;

    printf_e("\r\n");
    run_in_sram();
    return;
}
/**
  * @}
  */
/** @defgroup Base_Public_Functions Public functions
  * @brief    Base Public Functions
  * @{
  */
/**
  * @brief  Enter bootloader.
  * @retval None
  */
void boot_enter(ald_uart_handle_t *hperh)
{
    timer_init();
    boot_frame_init(hperh);

    shell_init(hperh);
    shell_cmd_insert("version", cmd_version, 0);
    shell_cmd_insert("reboot", cmd_reboot, 0);
    shell_cmd_insert("update", cmd_download, 0);
    shell_cmd_insert("run_flash", run_flash_cmd, 0);
    shell_cmd_insert("run_sram", run_sram_cmd, 0);

    if (!flash_usr_page_get())
    {
        printf_e("The APP flash is empty, Please Update first!\r\n");
    }
    else
    {
        if (check_app_crc())
        {
            printf_e("The APP flash is unempty, but verify failed.\r\n");
            printf_e("Please Update first\r\n");
        }
        else
        {
            printf_e("The APP flash is unempty, You can type the following cmd:\r\n");
            printf_e("update, run_flash, run_sram.\r\n");
            printf_e("Otherwise auto run after 3 seconds.\r\n");
            timer_start(3000, boot_timer_cbk);
        }
    }

    printf_e("\r\nes#");

    while (1)
    {
        if (shell_rx_flag_get())
        {
            shell_rx_flag_clear();
            shell_task_func(NULL);

            if (timer_env.state)
                timer_stop();
        }

        if (timer_env.timeout)
        {
            timer_env.timeout = 0;

            if (RUN_MODE_DEFAULT == RUN_FROM_FLASH)
                run_in_flash();
            else
                run_sram_cmd();
        }
    }
}
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
