/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for demo
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */ 

#include <string.h>
#include "main.h"

usr_uart_env_t env;
md_uart_init_t init;
uint8_t tx_buf[64];
uint8_t rx_buf[64];

/**
  * @brief  Init UART pin
  * @retval None
  */
void md_uart_pin_init(void)
{
	md_gpio_init_t x;
	/* Initialize tx pin */
	x.mode  = MD_GPIO_MODE_OUTPUT;
	x.odos  = MD_GPIO_PUSH_PULL;
	x.pupd  = MD_GPIO_PUSH_UP;
	x.podrv = MD_GPIO_OUT_DRIVE_1;
	x.nodrv = MD_GPIO_OUT_DRIVE_1;
	x.flt   = MD_GPIO_FILTER_DISABLE;
	x.type  = MD_GPIO_TYPE_TTL;
	x.func  = MD_GPIO_FUNC_3;
	md_gpio_init(UART0_TX_PORT, UART0_TX_PIN, &x);

	/* Initialize rx pin */
	x.mode  = MD_GPIO_MODE_INPUT;
	x.odos  = MD_GPIO_PUSH_PULL;
	x.pupd  = MD_GPIO_PUSH_UP;
	x.podrv = MD_GPIO_OUT_DRIVE_1;
	x.nodrv = MD_GPIO_OUT_DRIVE_1;
	x.flt   = MD_GPIO_FILTER_DISABLE;
	x.type  = MD_GPIO_TYPE_TTL;
	x.func  = MD_GPIO_FUNC_3;
	md_gpio_init(UART0_RX_PORT, UART0_RX_PIN, &x);
}

/**
  * @brief  Send data by interrupt
  * @param  buf: Pointer to data.
  * @param  size: Length of data.
  * @retval None
  */
void usr_uart_tx_by_interrupt(uint8_t *buf, uint32_t size)
{
	if ((size == 0) || (env.state & USR_STATE_TX))
		return;

	env.tx_buf  = buf;
	env.tx_cnt  = 0;
	env.tx_size = size;

	SET_BIT(env.state, USR_STATE_TX);
	md_uart_enable_it_tfempty(UARTx);
	return;
}

/**
  * @brief  Receive data by interrupt
  * @param  buf: Pointer to data.
  * @param  size: Length of data.
  * @retval None
  */
void usr_uart_rx_by_interrupt(uint8_t *buf, uint32_t size)
{
	if ((size == 0) || (env.state & USR_STATE_RX))
		return;

	env.rx_buf  = buf;
	env.rx_cnt  = 0;
	env.rx_size = size;

	SET_BIT(env.state, USR_STATE_RX);
	md_uart_set_rx_threshold(UARTx, 0x0);
	md_uart_enable_it_rfth(UARTx);
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	uint8_t i = 0;

	/* Configure system clock */
	md_cmu_pll1_config(MD_CMU_PLL1_INPUT_HOSC_3, MD_CMU_PLL1_OUTPUT_72M);
	md_cmu_clock_config(MD_CMU_CLOCK_PLL1, 72000000);
	/* PriorityGroup_2 */
	NVIC_SetPriorityGrouping(5); 
	/* Initialize SysTick Interrupt */
	md_init_1ms_tick();

	/* Enable ALL peripheral */
	SYSCFG_UNLOCK();
	md_cmu_enable_perh_all();
	SYSCFG_LOCK();

	/* Initialize uart communication gpio*/
	md_uart_pin_init();
	/* Initialize buffer */
	memset(rx_buf, 0x0, 64);
	for (i = 0; i < 64; ++i)
		tx_buf[i] = i + 1;

	md_uart_init_struct(&init);
	md_uart_init(UART0, &init);

	/* Enable interrupt */
	md_mcu_irq_config(UART0_IRQn, 2, 2, ENABLE);

	/* Send data by interrupt */
	usr_uart_tx_by_interrupt(tx_buf, 64);
	/* Receive data by interrupt */
	usr_uart_rx_by_interrupt(rx_buf, 20);

	while (1) {
		switch (env.event) {
		case USR_EVENT_TX_CPLT:
			/* TX completed */
			break;
		case USR_EVENT_RX_CPLT:
			/* RX completed */
			break;
		case USR_EVENT_TXRX_CPLT:
			/* TX and RX completed */
			break;
		case USR_EVENT_IDLE:
			break;
		default:
			break;
		}
		
		md_delay_1ms(10000);
		env.event = 0;

		usr_uart_tx_by_interrupt(tx_buf, 64);
		usr_uart_rx_by_interrupt(rx_buf, 20);
	}
}

