/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */ 

#include <string.h>
#include "main.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

//#define SPI_MASTER_ONLY_TX
#define SPI_MASTER_TX_RX

spi_handle_t h_spi;
uint8_t tx_buf[32];
uint8_t rx_buf[32];
volatile uint8_t only_tx_flag;
volatile uint8_t tx_rx_flag;
volatile uint32_t only_tx_cnt;
volatile uint32_t tx_rx_cnt;

extern dma_descriptor_t dma0_ctrl_base[28];

/**
  * @brief  Initializate pin of spi module.
  * @retval None
  */
void spi_pin_init(void)
{
	gpio_init_t x;

	/* Initialize nss pin */
	x.mode  = GPIO_MODE_OUTPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_1;
	x.nodrv = GPIO_OUT_DRIVE_1;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_TTL;
	x.func  = GPIO_FUNC_1;
	ald_gpio_init(GPIOC, GPIO_PIN_0, &x);

	/* Initialize sck pin */
	x.mode  = GPIO_MODE_OUTPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_1;
	x.nodrv = GPIO_OUT_DRIVE_1;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_TTL;
	x.func  = GPIO_FUNC_4;
	ald_gpio_init(GPIOC, GPIO_PIN_1, &x);

	/* Initialize miso pin */
	x.mode  = GPIO_MODE_INPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_1;
	x.nodrv = GPIO_OUT_DRIVE_1;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_TTL;
	x.func  = GPIO_FUNC_4;
	ald_gpio_init(GPIOC, GPIO_PIN_2, &x);

	/* Initialize mosi pin */
	x.mode  = GPIO_MODE_OUTPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_1;
	x.nodrv = GPIO_OUT_DRIVE_1;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_TTL;
	x.func  = GPIO_FUNC_4;
	ald_gpio_init(GPIOC, GPIO_PIN_3, &x);

	return;
}

/**
  * @brief  Initializate DMA configuration
  * @retval None
  */
void usr_spi_master_init(void)
{
	/* Configure SPI DMA transmit */
	h_spi.hdmatx.perh = DMA0;
	h_spi.hdmatx.cplt_arg = NULL;
	h_spi.hdmatx.cplt_cbk = NULL;
	h_spi.hdmatx.err_arg  = NULL;
	h_spi.hdmatx.err_cbk  = NULL;

	ald_dma_config_struct(&(h_spi.hdmatx.config));
	h_spi.hdmatx.config.data_width = DMA_DATA_SIZE_BYTE;
	h_spi.hdmatx.config.src        = (void *)0x20000000;
	h_spi.hdmatx.config.dst        = (void *)&h_spi.perh->DATA;
	h_spi.hdmatx.config.size       = 0;
	h_spi.hdmatx.config.src_inc    = DMA_DATA_INC_BYTE;
	h_spi.hdmatx.config.dst_inc    = DMA_DATA_INC_NONE;
	h_spi.hdmatx.config.msel       = DMA_MSEL_SPI1;
	h_spi.hdmatx.config.msigsel    = DMA_MSIGSEL_SPI_TXEMPTY;
	h_spi.hdmatx.config.burst      = ENABLE;
	h_spi.hdmatx.config.channel    = SPI_TX_DMA_CH;
	ald_dma_config_basic(&(h_spi.hdmatx));

#ifdef SPI_MASTER_TX_RX
	/* Configure DMA Receive */
	h_spi.hdmarx.perh = DMA0;
	h_spi.hdmarx.cplt_arg = NULL;
	h_spi.hdmarx.cplt_cbk = NULL;
	h_spi.hdmarx.err_arg  = NULL;
	h_spi.hdmarx.err_cbk  = NULL;

	ald_dma_config_struct(&(h_spi.hdmarx.config));
	h_spi.hdmarx.config.data_width = DMA_DATA_SIZE_BYTE;
	h_spi.hdmarx.config.src        = (void *)&h_spi.perh->DATA;
	h_spi.hdmarx.config.dst        = (void *)0x20000000;
	h_spi.hdmarx.config.size       = 0;
	h_spi.hdmarx.config.src_inc    = DMA_DATA_INC_NONE;
	h_spi.hdmarx.config.dst_inc    = DMA_DATA_INC_BYTE;
	h_spi.hdmarx.config.msel       = DMA_MSEL_SPI1;
	h_spi.hdmarx.config.msigsel    = DMA_MSIGSEL_SPI_RNR;
	h_spi.hdmarx.config.burst      = ENABLE;
	h_spi.hdmarx.config.channel    = SPI_RX_DMA_CH;
	ald_dma_config_basic(&(h_spi.hdmarx));
#endif
	return;
}

/**
  * @brief  Transmit an amount of data used DMA
  * @param  tx_buf: Pointer to data buffer
  * @param  size: Amount of data to be sent
  * @retval None
  */
void usr_spi_master_tx_start(uint8_t *tx_buf, uint32_t size)
{
	only_tx_flag = 0;

	dma0_ctrl_base[SPI_TX_DMA_CH].src  = (void *)(tx_buf + size - 1);
	dma0_ctrl_base[SPI_TX_DMA_CH].ctrl.n_minus_1 = size - 1;
	dma0_ctrl_base[SPI_TX_DMA_CH].ctrl.cycle_ctrl = DMA_CYCLE_CTRL_BASIC;

	DMA0->CHENSET = (0x1 << SPI_TX_DMA_CH);
	SET_BIT(h_spi.perh->CON2, SPI_CON2_TXDMA_MSK);

	return;
}

/**
  * @brief  Transmit and Receive an amount of data used DMA
  * @param  tx_buf: Pointer to data buffer
  * @param  rx_buf: Pointer to data buffer
  * @param  size: Amount of data to be sent
  * @retval None
  */
void usr_spi_master_tx_rx_start(uint8_t *tx_buf, uint8_t *rx_buf, uint32_t size)
{
	tx_rx_flag = 0;

	dma0_ctrl_base[SPI_TX_DMA_CH].src  = (void *)(tx_buf + size - 1);
	dma0_ctrl_base[SPI_TX_DMA_CH].ctrl.n_minus_1 = size - 1;
	dma0_ctrl_base[SPI_TX_DMA_CH].ctrl.cycle_ctrl = DMA_CYCLE_CTRL_BASIC;
	dma0_ctrl_base[SPI_RX_DMA_CH].dst = (void *)(rx_buf + size - 1);
	dma0_ctrl_base[SPI_RX_DMA_CH].ctrl.n_minus_1 = size - 1;
	dma0_ctrl_base[SPI_RX_DMA_CH].ctrl.cycle_ctrl = DMA_CYCLE_CTRL_BASIC;

	DMA0->CHENSET = (0x1 << SPI_TX_DMA_CH);
	DMA0->CHENSET = (0x1 << SPI_RX_DMA_CH);
	SET_BIT(h_spi.perh->CON2, SPI_CON2_TXDMA_MSK);
	SET_BIT(h_spi.perh->CON2, SPI_CON2_RXDMA_MSK);

	return;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	/* Initialize ALD */
	ald_cmu_init();
	/* Configure system clock */
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HRC_6, CMU_PLL1_OUTPUT_48M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);	
	ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);

	/* clear spi_handle_t structure */
	memset(&h_spi, 0x0, sizeof(h_spi));
	/* Initialize pin */
	spi_pin_init();
	/* Initialize tx_buf */
	memset(tx_buf, 0x55, sizeof(tx_buf));

	/* Initialize spi */
	h_spi.perh           = SPI1;
	h_spi.init.mode      = SPI_MODE_MASTER;
	h_spi.init.dir	     = SPI_DIRECTION_2LINES;
	h_spi.init.data_size = SPI_DATA_SIZE_8;
	h_spi.init.baud      = SPI_BAUD_64;
	h_spi.init.phase     = SPI_CPHA_SECOND;
	h_spi.init.polarity  = SPI_CPOL_HIGH;
	h_spi.init.first_bit = SPI_FIRSTBIT_MSB;
	h_spi.init.ss_en     = DISABLE;
	h_spi.init.crc_calc  = DISABLE;
	h_spi.tx_cplt_cbk    = NULL;
	h_spi.rx_cplt_cbk    = NULL;
	h_spi.err_cbk        = NULL;
	ald_spi_init(&h_spi);
	usr_spi_master_init();

	only_tx_cnt = 0;
	tx_rx_cnt   = 0;

	/* cs pin set low */
	ald_gpio_write_pin(GPIOC, GPIO_PIN_0, 0);

#ifdef SPI_MASTER_ONLY_TX
	usr_spi_master_tx_start(tx_buf, 32);
#endif

#ifdef SPI_MASTER_TX_RX
	usr_spi_master_tx_rx_start(tx_buf, rx_buf, 32);
#endif

	while (1) {
#ifdef SPI_MASTER_ONLY_TX
		if (only_tx_flag) {
			usr_spi_master_tx_start(tx_buf, 32);
			++only_tx_cnt;
		}
#endif
		
#ifdef SPI_MASTER_TX_RX
		if (tx_rx_flag) {
			usr_spi_master_tx_rx_start(tx_buf, rx_buf, 32);
			++tx_rx_cnt;
		}
#endif
	}
}

/**
  * @}
  */
/**
  * @}
  */
