/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    17 Nov 2017
  * @author  AE Team
  * @note   
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  *********************************************************************************
  */ 

#include <string.h>
#include "main.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

lcd_handle_t h_lcd;
uint32_t __buf[8] = {0};

void lcd_pin_init(void);
void lcd_init(lcd_handle_t *h_lcd);
void lcd_display_point(uint32_t *buf, uint8_t idx);
void lcd_display_numb(uint32_t *buf, uint8_t idx, uint8_t nb);
void display_all(uint32_t *buf);
void lcd_reflush(lcd_handle_t *hperh,uint32_t *buf);

/**
 * @brief 1~4 Digital Position COM And SEG MAP.
 */
const uint8_t __dig8[16]  = {
	0xED, 0x44, 0xD9, 0xD5, 0x74, 0xB5, 0xBD, 0xC4,
	0xFD, 0xF5, 0xFC, 0x3D, 0xA9, 0x5D, 0xB9, 0xB8
};

/**
 * @brief  LCD First Volume COM And SEG MAP.
 */
const uint32_t __map[8] = {
	(1 << 2), (1 << 1), (1 << 0), 0x80000000,
	(1 << 15), (1 << 14), (1 << 13), (1 << 12)
};

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	uint8_t i;

	/* Initialize ALD */
	ald_cmu_init();
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HRC_6, CMU_PLL1_OUTPUT_48M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);
	ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);
	lcd_init(&h_lcd);

	/* Display all */
	display_all(__buf);
	lcd_reflush(&h_lcd, __buf);
	memset(__buf, 0x00, 32);
	ald_delay_ms(2000);
	
	lcd_display_numb(__buf, 1, 0);
	lcd_reflush(&h_lcd, __buf);
	
	lcd_display_numb(__buf, 2, 0);
	lcd_reflush(&h_lcd, __buf);

	while (1) {
		/* Display number */
		for (i = 0; i < 16; ++i) {
			lcd_display_numb(__buf, 1, i);
			lcd_display_numb(__buf, 2, i);
			lcd_display_numb(__buf, 3, i);
			lcd_display_numb(__buf, 4, i);
			lcd_reflush(&h_lcd, __buf);

			memset(__buf, 0x00, 32);
			ald_delay_ms(1000);
		}

		/* Display point */
		for (i = 0; i < 4; ++i) {
			lcd_display_point(__buf, i + 1);
			lcd_reflush(&h_lcd, __buf);
			memset(__buf, 0x00, 32);
			ald_delay_ms(1000);
		}

		/* Flickering */
		lcd_display_numb(__buf, 1, 1);
		lcd_display_numb(__buf, 2, 2);
		lcd_display_numb(__buf, 3, 4);
		lcd_display_numb(__buf, 4, 8);
		lcd_display_point(__buf, 1);
		lcd_reflush(&h_lcd, __buf);
		memset(__buf, 0x00, 32);
		ald_lcd_blink_config(&h_lcd, LCD_BLINK_ALLSEG_ALLCOM, LCD_BLFRQ_512);
		ald_delay_ms(3000);
		ald_lcd_blink_config(&h_lcd, LCD_BLINK_OFF, LCD_BLFRQ_512);
	}
}


/**
  * @brief  Initializate lcd pin.
  * @note   	
  *      LCD  	      com1 com2 com3 com4 seg1  seg2  seg3  seg4  seg5  seg6  seg7  seg8
  *      MCU_LCD      com3 com2 com1 com0 seg36 seg37 seg0 seg11 seg12 seg15 seg14 seg13
  *      3366LT       com7 com6 com5 com0 seg34 seg33 seg32 seg31 seg15 seg14 seg13 seg12            
  * @retval None
  */
void lcd_pin_init(void)
{
	gpio_init_t x;	

	/* MCU_COM7--->LCD_COM1 */
	x.mode = GPIO_MODE_CLOSE;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_FLOATING;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_CMOS;
	x.func = GPIO_FUNC_0;
	ald_gpio_init(GPIOA, GPIO_PIN_10, &x);

	/* MCU_COM6--->LCD_COM2 */
	x.mode = GPIO_MODE_CLOSE;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_FLOATING;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_CMOS;
	x.func = GPIO_FUNC_0;
	ald_gpio_init(GPIOA, GPIO_PIN_9, &x);

	/* MCU_COM5--->LCD_COM3 */
	x.mode = GPIO_MODE_CLOSE;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_FLOATING;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_CMOS;
	x.func = GPIO_FUNC_0;
	ald_gpio_init(GPIOA, GPIO_PIN_8, &x);

	/* MCU_COM0--->LCD_COM4 */
	x.mode = GPIO_MODE_CLOSE;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_FLOATING;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_CMOS;
	x.func = GPIO_FUNC_0;
	ald_gpio_init(GPIOD, GPIO_PIN_15, &x);

	/* MCU_SEG34--->SEG1 */
	x.mode = GPIO_MODE_CLOSE;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_FLOATING;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_CMOS;
	x.func = GPIO_FUNC_0;
	ald_gpio_init(GPIOD, GPIO_PIN_14, &x);

	/* MCU_SEG33--->SEG2 */
	x.mode = GPIO_MODE_CLOSE;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_FLOATING;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_CMOS;
	x.func = GPIO_FUNC_0;
	ald_gpio_init(GPIOD, GPIO_PIN_13, &x);

	/* MCU_SEG32--->SEG3 */
	x.mode = GPIO_MODE_CLOSE;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_FLOATING;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_CMOS;
	x.func = GPIO_FUNC_0;
	ald_gpio_init(GPIOD, GPIO_PIN_12, &x);

	/*MCU_SEG31--->SEG4 */
	x.mode = GPIO_MODE_CLOSE;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_FLOATING;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_CMOS;
	x.func = GPIO_FUNC_0;
	ald_gpio_init(GPIOD, GPIO_PIN_11, &x);

	/*MCU_SEG15--->SEG5 */
	x.mode = GPIO_MODE_CLOSE;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_FLOATING;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_CMOS;
	x.func = GPIO_FUNC_0;
	ald_gpio_init(GPIOB, GPIO_PIN_15, &x);

	/*MCU_SEG14--->SEG6 */
	x.mode = GPIO_MODE_CLOSE;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_FLOATING;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_CMOS;
	x.func = GPIO_FUNC_0;
	ald_gpio_init(GPIOB, GPIO_PIN_14, &x);

	/*MCU_SEG13--->SEG7 */
	x.mode = GPIO_MODE_CLOSE;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_FLOATING;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_CMOS;
	x.func = GPIO_FUNC_0;
	ald_gpio_init(GPIOB, GPIO_PIN_13, &x);

	/*MCU_SEG12--->SEG8 */
	x.mode = GPIO_MODE_CLOSE;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_FLOATING;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_CMOS;
	x.func = GPIO_FUNC_0;
	ald_gpio_init(GPIOB, GPIO_PIN_12, &x);

	return;
}

/**
  * @brief  Initializate lcd.
  * @param  h_lcd: Pointer to LCD controler.
  * @retval None
  */
void lcd_init(lcd_handle_t *h_lcd)
{
	lcd_pin_init();

	/* Initialize lcd */
	h_lcd->perh = LCD;
	h_lcd->init.lcd_bias   = LCD_BIAS_1_3;
	h_lcd->init.lcd_div    = LCD_DIV_27;
	h_lcd->init.lcd_duty   = LCD_DUTY_1_8;
	h_lcd->init.lcd_prs    = LCD_PRS_2;
	h_lcd->init.lcd_reshd  = LCD_RES_3MOHM;
	h_lcd->init.lcd_resld  = LCD_RES_3MOHM;
	h_lcd->init.lcd_vbufhd = LCD_FUNC_ENABLE;
	h_lcd->init.lcd_vbufld = LCD_FUNC_DISABLE;
	h_lcd->init.lcd_vchps  = LCD_VCHPS_3V2;
	h_lcd->init.lcd_vsel   = LCD_VSEL_VDD;
	h_lcd->init.lcd_wfs    = LCD_WAVE_B;
	h_lcd->init.lcd_vgs    = LCD_VGS_15;
	h_lcd->init.clock      = CMU_LCD_SEL_LRC;
	ald_lcd_init(h_lcd);

	/* seg34 seg33 seg32 seg31 seg15 seg14 seg13 seg12 */
	ald_lcd_write_seg(h_lcd, SEG_0_TO_31, 0x8000F000);
	ald_lcd_write_seg(h_lcd, SEG_32_TO_59, 0x00000007);

	/* Enable lcd */
	ald_lcd_cmd(h_lcd, ENABLE);
}

/**
  * @brief  LCD fill display buffer.
  * @param  buf: buffer start address.
  * @param  idx: display point index
  * @retval None.
  */
void lcd_display_point(uint32_t *buf, uint8_t idx)
{
	switch (idx) {
	case 1:
		buf[1] |= __map[0];
		break;
	case 2:
		buf[0] |= __map[2];
		break;
	case 3:
		buf[0] |= __map[4];
		break;
	case 4:
		buf[0] |= __map[6];
		break;
	default:
		break;
	}

	return;
}

/**
  * @brief  LCD fill display buffer.
  * @param  buf: buffer start address.
  * @param  idx: display digital index
  * @param  nb: number 0x0 ~ 0xF
  * @retval None.
  */
void lcd_display_numb(uint32_t *buf, uint8_t idx, uint8_t nb)
{
	uint8_t i, tmp;
	uint8_t msk = 0x3;

	if (idx > 4 || idx < 1)
		return;

	for (i = 0; i < 4; ++i) {				
		tmp = (__dig8[nb] & msk) >> 2 * i;

		switch (tmp) {
		case 1:
			if (idx == 1)
				buf[1 + 2 * i] |=  __map[idx * 2 - 1];	
			else
				buf[2*i] |= __map[idx * 2 - 1];
			break;
		case 2:
			if (idx == 1)
				buf[1 + 2*i] |= __map[idx * 2 - 2];
			else if (idx == 2) 
				buf[1 + 2*i] |= __map[idx * 2 - 2];
			else 
				buf[2*i] |= __map[idx * 2 - 2];
			break;
		case 3:
			if (idx == 1)
				buf[1 + 2 * i] |= (__map[idx * 2 - 1] | __map[idx * 2 - 2]);
			else if (idx == 2) {
				buf[1 + 2 * i] |= __map[idx * 2 - 2];
				buf[2*i] |= __map[idx * 2 - 1];
			} else
				buf[2*i] |= (__map[idx * 2 - 1] | __map[idx * 2 - 2]);
			break;
		default:
			break;
		}
		msk <<= 2;
	}
}

/**
  * @brief  Display all.
  * @param  buf: buffer start address.
  * @retval None.
  */
void display_all(uint32_t *buf)
{
	lcd_display_numb(__buf, 1, 8);
	lcd_display_numb(__buf, 2, 8);
	lcd_display_numb(__buf, 3, 8);
	lcd_display_numb(__buf, 4, 8);
	lcd_display_point(__buf, 1);
	lcd_display_point(__buf, 2);
	lcd_display_point(__buf, 3);
	lcd_display_point(__buf, 4);
}	


/**
  * @brief  Reflush the used com_seg buffer.
  * @param  hperh: Pointer to LCD controler.
  * @param  buf: Buffer to reflush.
  * @retval None.
  */
void lcd_reflush(lcd_handle_t *hperh,uint32_t *buf)
{
	/* MCU_COM7---> LCD_COM1 */
	ald_lcd_write(hperh, 14, buf[6]);//
	ald_lcd_write(hperh, 15, buf[7]);//
	/* MCU_COM6---> LCD_COM2 */
	ald_lcd_write(hperh, 12, buf[4]);
	ald_lcd_write(hperh, 13, buf[5]);
	/* MCU_COM5---> LCD_COM3 */
	ald_lcd_write(hperh, 10, buf[2]);
	ald_lcd_write(hperh, 11, buf[3]);//
	/* MCU_COM0---> LCD_COM4 */
	ald_lcd_write(hperh, 0, buf[0]);
	ald_lcd_write(hperh, 1, buf[1]);
}

/**
  * @}
  */
/**
  * @}
  */
