/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include <string.h>
#include "main.h"



/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

uint32_t __value;
uint8_t __flag;
dac_handle_t h_dac;
dac_channel_config_t channel_config;

/**
  * @brief  DAC gpio init.
  * @retval None.
  */
void dac_gpio_init(void)
{
	gpio_init_t x;
	memset(&x, 0, sizeof(gpio_init_t));

	x.mode = GPIO_MODE_OUTPUT;
	x.func = GPIO_FUNC_0;
	ald_gpio_init(DAC0_OUT_PORT, DAC0_OUT_PIN, &x);
	return;
}

/**
  * @brief  DAC event callback.
  * @param  arg: Pointer to dac_handle_t structure.
  *              @arg DAC_EVENT_CH0_CPLT - Channel 0 completed
  *              @arg DAC_EVENT_CH1_CPLT - Channel 1 completed
  *              @arg DAC_EVENT_CH0_UF - Channel 0 under flow
  *              @arg DAC_EVENT_CH1_UF - Channel 1 under flow
  * @retval None.
  */
void dac_cbk(dac_handle_t *arg, uint32_t event)
{
	switch (event) {
	case DAC_EVENT_CH0_CPLT:
		__flag = 1;
		break;
	case DAC_EVENT_CH1_CPLT:
		break;
	case DAC_EVENT_CH0_UF:
		break;
	case DAC_EVENT_CH1_UF:
		break;
	default:
		break;
	}

	return;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	/* Initialize ALD */
	ald_cmu_init();
	/* Configure system clock */
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_72M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 72000000);

	/* Enable mode clock */
	ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);
	/* DAC pin Initialize */
	dac_gpio_init();
	/* Enable dac0 ch0 interrupt */
	ald_mcu_irq_config(DAC0_CH0_IRQn, 1, 1, ENABLE);

	memset(&h_dac, 0, sizeof(dac_handle_t));
	memset(&channel_config, 0, sizeof(dac_channel_config_t));

	/* Initialize DAC */
	h_dac.perh            = DAC0;
	h_dac.init.conv_mode  = DAC_CONV_MODE_CONTINUOUS;
	h_dac.init.out_mode   = DAC_OUTPUT_PIN;
	h_dac.init.refresh    = DAC_REFRESH_8;
	h_dac.init.div        = DAC_PRES_DIV_64;
	h_dac.init.ch0_reset  = DISABLE;
	h_dac.init.o_ctrl_pis = DISABLE;
	h_dac.init.sine       = DISABLE;
	h_dac.init.diff       = DISABLE;
	h_dac.init.n_ref      = DAC_NEG_REF_VSS;
	h_dac.init.p_ref      = DAC_POS_REF_VDD;
	h_dac.cbk             = dac_cbk;
	ald_dac_init(&h_dac);

	/* Config dac channel 0 */
	channel_config.enable     = ENABLE;
	channel_config.trigger    = DAC_TRIGGER_BY_DATA;
	channel_config.refresh_en = DISABLE;
	ald_dac_channel_config(&h_dac, &channel_config, DAC_CHANNEL_0);
	ald_dac_interrupt_config(&h_dac, DAC_IT_CH0, ENABLE);

	/* Set channel 0 output value */
	__value = 4095;
	__flag  = 0;
	ald_dac_output_set(&h_dac, DAC_CHANNEL_0, __value);
	while (!__flag);

	while (1) {
		ald_delay_ms(2000);

		__value  = __value < 200 ? 4095 : __value;
		__value -= 200;
		__flag   = 0;

		ald_dac_output_set(&h_dac, DAC_CHANNEL_0, __value);
		while (!__flag);
	}
}

/**
  * @}
  */
/**
  * @}
  */
