/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include <string.h>
#include "main.h"


/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

crypt_handle_t h_crypt;
uint8_t count = 0;
uint8_t error = 0;

/**
  * @brief  crypt completed callback.
  * @param  arg: Pointer to crypt_handle_t structure.
  * @retval None.
  */
void crypt_complete(crypt_handle_t *arg)
{
	ald_crypt_clear_flag_status(arg, CRYPT_FLAG_DONE);
	++count;
	return;
}

/**
  * @brief  crypt error callback.
  * @param  arg: Pointer to crypt_handle_t structure.
  * @retval None.
  */
void error_complete(crypt_handle_t *arg)
{
	++error;
	return;
}

/* aes ecb key = 128bits
 * input 	32 43 f6 a8 88 5a 30 8d 31 31 98 a2 e0 37 07 34
 * cipher key	2b 7e 15 16 28 ae d2 a6 ab f7 15 88 09 cf 4f 3c
 * refer result	39 25 84 1d 02 dc 09 fb dc 11 85 97 19 6a 0b 32
 */
void test_aes_ecb_128()
{
	uint32_t plain_buffer[4] = {0x3243F6A8, 0x885A308D, 0x313198A2, 0xE0370734};
	uint32_t result_buffer[4] = {0};

	uint32_t expect_buffer[4] = {0x3925841d, 0x02dc09fb, 0xdc118597, 0x196a0b32};

	uint32_t key_buf[4]   = {0x2b7e1516, 0x28aed2a6, 0xabf71588, 0x09cf4f3c};

	/* Clear lpuart_handle_t structure */
	memset(&h_crypt, 0x0, sizeof(crypt_handle_t));

	/* Initialize crypt */
	h_crypt.perh           = CRYPT;
	h_crypt.init.mode      = CRYPT_MODE_ECB;//CRYPT_MODE_ECB
	h_crypt.init.key       = CRYPT_AES_BITS_128;
	h_crypt.init.type      = DATA_32_BIT;
	h_crypt.crypt_cplt_cbk = crypt_complete;
	h_crypt.err_cplt_cbk   = error_complete;
	ald_crypt_init(&h_crypt);

	/* Start AES encrypt,right result_buffer equal to expect_buffer */
	ald_crypt_write_key(&h_crypt,key_buf,KEY_4_LEN);
 	ald_crypt_encrypt(&h_crypt,(uint8_t*)plain_buffer, (uint8_t*)result_buffer, sizeof(plain_buffer));

	/* Start AES decrypt,right result_buffer equal to plain_buffer */	
	ald_crypt_write_key(&h_crypt,key_buf,KEY_4_LEN);
 	ald_crypt_decrypt(&h_crypt,(uint8_t*)expect_buffer, (uint8_t*)result_buffer, sizeof(expect_buffer));	
}


/* aes ecb key = 192bits 
 * input	00 11 22 33 44 55 66 77 88 99 aa bb cc dd ee ff
 * key		00 01 02 03 04 05 06 07 08 09 0a 0b 0c 0d 0e 0f 10 11 12 13 14 15 16 17
 * refer result dd a9 7c a4 86 4c df e0 6e af 70 a0 ec 0d 71 91
 */
void test_aes_ecb_192()
{
	uint32_t plain_buffer[4] = {0x00112233, 0x44556677, 0x8899aabb, 0xccddeeff};
	uint32_t result_buffer[4] = {0};

	uint32_t expect_buffer[4] = {0xdda97ca4, 0x864cdfe0, 0x6eaf70a0, 0xec0d7191};

	uint32_t key_buf[6]   = {0x00010203, 0x04050607, 0x08090a0b, 0x0c0d0e0f, 0x10111213, 0x14151617};

	/* Clear lpuart_handle_t structure */
	memset(&h_crypt, 0x0, sizeof(crypt_handle_t));

	/* Initialize crypt */
	h_crypt.perh           = CRYPT;
	h_crypt.init.mode      = CRYPT_MODE_ECB;//CRYPT_MODE_ECB
	h_crypt.init.key       = CRYPT_AES_BITS_192;
	h_crypt.init.type      = DATA_32_BIT;
	h_crypt.crypt_cplt_cbk = crypt_complete;
	h_crypt.err_cplt_cbk   = error_complete;
	ald_crypt_init(&h_crypt);

	/* Start AES encrypt,right result_buffer equal to expect_buffer */
	ald_crypt_write_key(&h_crypt,key_buf,KEY_6_LEN);
 	ald_crypt_encrypt(&h_crypt,(uint8_t*)plain_buffer, (uint8_t*)result_buffer, sizeof(plain_buffer));

	/* Start AES decrypt,right result_buffer equal to plain_buffer */	
	ald_crypt_write_key(&h_crypt,key_buf,KEY_6_LEN);
 	ald_crypt_decrypt(&h_crypt,(uint8_t*)expect_buffer, (uint8_t*)result_buffer, sizeof(expect_buffer));	
}

/* aes ecb key = 256bits 
 * input	00 11 22 33 44 55 66 77 88 99 aa bb cc dd ee ff
 * key		00 01 02 03 04 05 06 07 08 09 0a 0b 0c 0d 0e 0f 10 11 12 13 14 15 16 17 18 19 1a 1b 1c 1d 1e 1f
 * refer result 8e a2 b7 ca 51 67 45 bf ea fc 49 90 4b 49 60 89
 */
void test_aes_ecb_256()
{
	uint32_t plain_buffer[4] = {0x00112233, 0x44556677, 0x8899aabb, 0xccddeeff};
	uint32_t result_buffer[4] = {0};

	uint32_t expect_buffer[4] = {0x8ea2b7ca, 0x516745bf, 0xeafc4990, 0x4b496089};

	uint32_t key_buf[8]   = {	0x00010203, 0x04050607, 0x08090a0b, 0x0c0d0e0f,
					0x10111213, 0x14151617, 0x18191a1b, 0x1c1d1e1f};

	/* Clear lpuart_handle_t structure */
	memset(&h_crypt, 0x0, sizeof(crypt_handle_t));

	/* Initialize crypt */
	h_crypt.perh           = CRYPT;
	h_crypt.init.mode      = CRYPT_MODE_ECB;//CRYPT_MODE_ECB
	h_crypt.init.key       = CRYPT_AES_BITS_256;
	h_crypt.init.type      = DATA_32_BIT;
	h_crypt.crypt_cplt_cbk = crypt_complete;
	h_crypt.err_cplt_cbk   = error_complete;
	ald_crypt_init(&h_crypt);

	/* Start AES encrypt,right result_buffer equal to expect_buffer */
	ald_crypt_write_key(&h_crypt,key_buf,KEY_8_LEN);
 	ald_crypt_encrypt(&h_crypt,(uint8_t*)plain_buffer, (uint8_t*)result_buffer, sizeof(plain_buffer));

	/* Start AES decrypt,right result_buffer equal to plain_buffer */	
	ald_crypt_write_key(&h_crypt,key_buf,KEY_8_LEN);
 	ald_crypt_decrypt(&h_crypt,(uint8_t*)expect_buffer, (uint8_t*)result_buffer, sizeof(expect_buffer));	
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	/* Initialize ALD */
	ald_cmu_init();
	/* Configure system clock */
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HRC_6, CMU_PLL1_OUTPUT_48M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);
	/* Enable CRC mode clock */
	ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);

	/* Aes ebc mode 128bits key case */
	test_aes_ecb_128();
	/* Aes ebc mode 192bits key case */
	test_aes_ecb_192();
	/* Aes ebc mode 256bits key case */
	test_aes_ecb_256();

	while (1) {
		ald_delay_ms(1000);
	}
}

/**
  * @}
  */
/**
  * @}
  */
