/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */


#include "main.h"
#include <stdarg.h>
#include <stdio.h>
#include <string.h>

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

can_handle_t h_can;
can_filter_t filter0;
can_filter_t filter1;
can_tx_msg_t tx_msg;
can_rx_msg_t rx0_msg;
can_rx_msg_t rx1_msg;
uint32_t FID0 = 0x486;
uint32_t FID1 = 0x123;

/**
  * @brief  Initializate pin of can module.
  * @retval None
  */
void can_pin_init(void)
{
	gpio_init_t x;

	/* Initialize tx pin */
	x.mode  = GPIO_MODE_OUTPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_TTL;
	x.func  = GPIO_FUNC_3;
	ald_gpio_init(CAN_TX_PORT, CAN_TX_PIN, &x);

	/* Initialize rx pin */
	x.mode  = GPIO_MODE_INPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_TTL;
	x.func  = GPIO_FUNC_3;
	ald_gpio_init(CAN_RX_PORT, CAN_RX_PIN, &x);

	return;
}

/**
  * @brief  Send a message using default parameters.
  * @id    arg: Can frame id
  * @data  arg: Can frame data
  * @retval HAL status.
  */
ald_status_t can_send_msg(uint32_t id, uint8_t data)
{
#ifdef  EXTEND_ID
	tx_msg.type = CAN_ID_EXT;
	tx_msg.ext  = id;
#else
	tx_msg.type = CAN_ID_STD;
	tx_msg.std  = id;
#endif
	tx_msg.rtr  = CAN_RTR_DATA;
	tx_msg.len  = 8;
	memset(tx_msg.data, data, tx_msg.len);

	return ald_can_send(&h_can, &tx_msg, 100);
}

/**
  * @brief  Send message complete.
  * @param  arg: Pointer to can_handle_t structure.
  * @retval None.
  */
void can_send_complete(can_handle_t *arg)
{
	return;
}

/**
  * @brief  Receive a message complete.
  * @param  arg: Pointer to can_handle_t structure.
  * @param  num: Index of the RxFIFO.
  * @retval None.
  */
void can_recv_complete(can_handle_t *arg, can_rx_fifo_t num)
{
	ald_can_recv_by_it(&h_can, num, num == CAN_RX_FIFO0 ? &rx0_msg : &rx1_msg);	
	return;
}

/**
  * @brief  Occurs error.
  * @param  arg: Pointer to can_handle_t structure.
  * @retval None.
  */
void can_error(can_handle_t *arg)
{
	return;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	/* Initialize ALD */
	ald_cmu_init();

	/* Configure system clock */
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_48M);
	ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);

	/* Enable CAN interrupt */
	ald_mcu_irq_config(CAN0_RX0_IRQn, 3, 3,ENABLE);
        ald_mcu_irq_config(CAN0_RX1_IRQn, 3, 3,ENABLE);
	ald_mcu_irq_config(CAN0_TX_IRQn, 3, 3,ENABLE);

	/* clear can_handle_t structure */
	memset(&h_can, 0x0, sizeof(h_can));
	/* clear can_filter_t structure */
	memset(&filter0, 0x0, sizeof(filter0));
	memset(&filter1, 0x0, sizeof(filter1));
	/* Initialize pin */
	can_pin_init();

	/* Initialize CAN */
	h_can.perh        = CAN0;
	h_can.init.psc    = 8;
	h_can.init.mode   = CAN_MODE_NORMAL;
	h_can.init.sjw    = CAN_SJW_1;
	h_can.init.seg1   = CAN_SEG1_7;
	h_can.init.seg2   = CAN_SEG2_4;
	h_can.init.ttcm   = DISABLE;
	h_can.init.abom   = DISABLE;
	h_can.init.awk    = DISABLE;
	h_can.init.artx   = ENABLE;
	h_can.init.rfom   = DISABLE;
	h_can.init.txmp   = DISABLE;
	h_can.tx_cplt_cbk = can_send_complete;
	h_can.rx_cplt_cbk = can_recv_complete;
	h_can.error_cbk   = can_error;
	ald_can_init(&h_can);	

	filter0.id_high      = FID0 << 5;
	filter0.id_low       = 0;
	filter0.mask_id_high = 0xFFFF;
	filter0.mask_id_low  = 0xFFFF;
	filter0.fifo         = CAN_FILTER_FIFO0;
	filter0.number       = 0;
	filter0.mode         = CAN_FILTER_MODE_MASK;
	filter0.scale        = CAN_FILTER_SCALE_32;	
	filter0.active       = ENABLE;
	ald_can_filter_config(&h_can, &filter0);

	filter1.id_high      = FID1 << 5;
	filter1.id_low       = 0;
	filter1.mask_id_high = 0xFFFF;
	filter1.mask_id_low  = 0xFFFF;
	filter1.fifo         = CAN_FILTER_FIFO1;
	filter1.number       = 1;
	filter1.mode         = CAN_FILTER_MODE_MASK;
	filter1.scale        = CAN_FILTER_SCALE_32;	
	filter1.active       = ENABLE;
	ald_can_filter_config(&h_can, &filter1);

	/* Receive a message */
	ald_can_recv_by_it(&h_can, CAN_RX_FIFO0, &rx0_msg);
	ald_can_recv_by_it(&h_can, CAN_RX_FIFO1, &rx1_msg);

	while (1) {		
		ald_delay_ms(1000);
	}
}
/**
  * @}
  */
/**
  * @}
  */
