/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include "main.h"


/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */


adc_handle_t h_adc;
uart_handle_t h_uart;
pis_handle_t h_pis;
timer_handle_t h_timer;
char uart_buf[64];
env_t _env;

void get_normal_average_voltage(void);

/**
  * @brief  Output debug information via UART.
  * @param  fmt: Varibale parameter
  * @retval None
  */
void printf_e(const char *fmt, ...)
{
	va_list args;

	va_start(args, fmt);
	vsnprintf(uart_buf, 64, fmt, args);
	va_end(args);

	ald_uart_send(&h_uart, (uint8_t *)uart_buf, strlen(uart_buf), 1000);
	return;
}

/**
  * @brief  Initialize the UART
  * @retval None
  */
void uart_stdio_init(void)
{
	gpio_init_t x;

	/* Initialize tx pin */
	x.mode  = GPIO_MODE_OUTPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_TTL;
	x.func  = UART_FUNC_TX;
	ald_gpio_init(UART_PORT_TX, UART_PIN_TX, &x);

	/* Initialize rx pin */
	x.mode  = GPIO_MODE_INPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_TTL;
	x.func  = UART_FUNC_RX;
	ald_gpio_init(UART_PORT_RX, UART_PIN_RX, &x);

	/* Initialize uart */
	h_uart.perh             = UART0;
	h_uart.init.baud        = 115200;
	h_uart.init.word_length = UART_WORD_LENGTH_8B;
	h_uart.init.stop_bits   = UART_STOP_BITS_1;
	h_uart.init.parity      = UART_PARITY_NONE;
	h_uart.init.mode        = UART_MODE_UART;
	h_uart.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
	h_uart.tx_cplt_cbk      = NULL;
	h_uart.rx_cplt_cbk      = NULL;
	h_uart.error_cbk        = NULL;
	ald_uart_init(&h_uart);

	return;
}

/**
  * @brief  Adc normal convert complete.
  * @param  arg: Pointer to adc_handle_t structure.
  * @retval None.
  */
static void normal_complete_cbk(adc_handle_t *arg)
{
	return;
}

/**
  * @brief  Adc insert convert complete.
  * @param  arg: Pointer to adc_handle_t structure.
  * @retval None.
  */
static void insert_complete_cbk(adc_handle_t *arg)
{
	_env.buf[0][_env.idx] = h_adc.perh->ICHDR1;
	_env.buf[1][_env.idx] = h_adc.perh->ICHDR2;
	_env.buf[2][_env.idx] = h_adc.perh->ICHDR3;
	_env.buf[3][_env.idx] = h_adc.perh->ICHDR4;

	if (++_env.idx >= USR_ADC_COUNT) {
		ald_timer_base_stop(&h_timer);
		ADC_DISABLE(&h_adc);
		_env.flag = 1;
	}

	return;
}

/**
  * @brief  Adc level out of window complete.
  * @param  arg: Pointer to adc_handle_t structure.
  * @retval None.
  */
static void wdg_cbk(adc_handle_t *arg)
{
	return;
}

/**
  * @brief  Adc error complete.
  * @param  arg: Pointer to adc_handle_t structure.
  * @retval None.
  */
static void error_cbk(adc_handle_t *arg)
{
	return;
}

/**
  * @brief  Adc over complete.
  * @param  arg: Pointer to adc_handle_t structure.
  * @retval None.
  */
static void ovr_cbk(adc_handle_t *arg)
{
	return;
}

/**
  * @brief  Initializate pin of adc module.
  * @retval None.
  */
void adc_pin_init(void)
{
	gpio_init_t x;

	memset(&x, 0, sizeof(gpio_init_t));

	/* Initialize ADC pin */
	x.mode  = GPIO_MODE_INPUT;
	x.pupd  = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_CMOS;
	x.func  = GPIO_FUNC_0;

	ald_gpio_init(GPIO_CH0_PORT, ADC_CH0_PIN, &x);
	ald_gpio_init(GPIO_CH2_PORT, ADC_CH2_PIN, &x);
	ald_gpio_init(GPIO_CH3_PORT, ADC_CH3_PIN, &x);
	ald_gpio_init(GPIO_CH11_PORT, ADC_CH11_PIN, &x);

	return;
}

/**
  * @brief  Initializate ADC.
  * @retval None.
  */
void adc_init(void)
{
	adc_ich_conf_t i_conf;

	/* clear structures */
	memset(&h_adc, 0x0, sizeof(adc_handle_t));
	memset(&h_pis, 0x0, sizeof(pis_handle_t));
	memset(&h_timer, 0x0, sizeof(timer_handle_t));
	memset(&i_conf, 0x0, sizeof(adc_ich_conf_t));

	/* Initialize PIS */
	h_pis.perh               = PIS;
	h_pis.init.producer_clk  = PIS_CLK_PCLK1;
	h_pis.init.producer_edge = PIS_EDGE_NONE;
	h_pis.init.consumer_clk  = PIS_CLK_PCLK2;
	h_pis.init.producer_src  = PIS_TIMER1_UPDATA;
	h_pis.init.consumer_trig = PIS_CH7_ADC0_INSERT;
	ald_pis_create(&h_pis);

	/* Initialize TIMER1 */
	h_timer.perh           = GP16C4T1;
	h_timer.init.prescaler = 0;
	h_timer.init.mode      = TIMER_CNT_MODE_UP;
	h_timer.init.period    = ((ald_cmu_get_pclk1_clock() / 1000000) * 200);
	h_timer.init.clk_div   = TIMER_CLOCK_DIV1;
	h_timer.init.re_cnt    = 0;
	ald_timer_base_init(&h_timer);	

	/* Initialize adc */
	h_adc.perh            = ADC0;
	h_adc.init.align      = ADC_DATAALIGN_RIGHT;
	h_adc.init.scan       = ENABLE;
	h_adc.init.cont       = DISABLE;
	h_adc.init.nch_nr     = ADC_NCH_NR_1;
	h_adc.init.disc       = ADC_ALL_DISABLE;
	h_adc.init.disc_nr    = ADC_DISC_NR_1;
	h_adc.init.data_bit   = ADC_CONV_BIT_12;
	h_adc.init.div        = ADC_CKDIV_128;
	h_adc.init.nche_sel   = ADC_NCHESEL_MODE_ALL;
	h_adc.init.n_ref      = ADC_NEG_REF_VSS;
	h_adc.init.p_ref      = ADC_POS_REF_VDD;
	h_adc.normal_cplt_cbk = normal_complete_cbk;
	h_adc.insert_cplt_cbk = insert_complete_cbk;
	h_adc.wdg_cbk         = wdg_cbk;
	h_adc.error_cbk       = error_cbk;
	h_adc.ovr_cbk         = ovr_cbk;
	ald_adc_init(&h_adc);
	ald_adc_interrupt_config(&h_adc, ADC_IT_ICH, ENABLE);
	ADC_IETS_ENABLE(&h_adc, ADC_ETS_RISE);

	/* Initialize insert convert channel */
	i_conf.ch     = ADC_CHANNEL_0;
	i_conf.idx    = ADC_ICH_IDX_1;
	i_conf.samp   = ADC_SAMPLETIME_4;
	i_conf.offset = 0;
	i_conf.nr     = ADC_ICH_NR_4;
	i_conf.auto_m = DISABLE;
	ald_adc_insert_channel_config(&h_adc, &i_conf);

	i_conf.ch     = ADC_CHANNEL_2;
	i_conf.idx    = ADC_ICH_IDX_2;
	i_conf.samp   = ADC_SAMPLETIME_4;
	i_conf.offset = 0;
	i_conf.nr     = ADC_ICH_NR_4;
	i_conf.auto_m = DISABLE;
	ald_adc_insert_channel_config(&h_adc, &i_conf);

	i_conf.ch     = ADC_CHANNEL_3;
	i_conf.idx    = ADC_ICH_IDX_3;
	i_conf.samp   = ADC_SAMPLETIME_4;
	i_conf.offset = 0;
	i_conf.nr     = ADC_ICH_NR_4;
	i_conf.auto_m = DISABLE;
	ald_adc_insert_channel_config(&h_adc, &i_conf);
	
	i_conf.ch     = ADC_CHANNEL_11;
	i_conf.idx    = ADC_ICH_IDX_4;
	i_conf.samp   = ADC_SAMPLETIME_4;
	i_conf.offset = 0;
	i_conf.nr     = ADC_ICH_NR_4;
	i_conf.auto_m = DISABLE;
	ald_adc_insert_channel_config(&h_adc, &i_conf);

	return;
}

/**
  * @brief  Get the average value of the normal channel.
  * @retval None
  */
void usr_adc_insert_start(void)
{
	/* Enable ADC peripheral */
	ADC_ENABLE(&h_adc);
	/* Start insert convert */
	ald_timer_base_start(&h_timer);
	return;
}

/**
  * @brief  Get average value of insert channel
  * @retval None.
  */
void usr_adc_average_value_get(void)
{
	int i, k;
	int sum, min, max;

	for (k = 0; k < USR_ADC_CH_NR; ++k) {
		sum = 0;
		min = 0xFFF;
		max = 0;

		for (i = 0; i < USR_ADC_COUNT; ++i) {
			min = _env.buf[k][i] < min ? _env.buf[k][i] : min;
			max = _env.buf[k][i] > max ? _env.buf[k][i] : max;
			sum += _env.buf[k][i];
		}

		_env.avg[k] = (sum - max - min) / (USR_ADC_COUNT - 2);
		_env.avg[k] = (_env.avg[k] * USR_VDD_VALUE) >> 12;
	}

	return;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	/* Initialize ALD */
	ald_cmu_init();
	/* Configure system clock */
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_48M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);
	/* Enable peripheral clock */
	ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);
	/* Enable adc interrupt */
	ald_mcu_irq_config(ADC0_IRQn, 1, 1, ENABLE);
	/* Clear env_t structure */
	memset(&_env, 0x0, sizeof(env_t));

	/* Initialize stdio */
	uart_stdio_init();
	/* Initialize ADC pin */
	adc_pin_init();
	/* Initialize ADC */
	adc_init();
	/* Start insert channel */
	usr_adc_insert_start();

	printf_e("System start...\r\n");
	printf_e("Index CH0    CH2    CH3    CH11\r\n");
	printf_e("----------------------------------------------\r\n");
	while (!_env.flag);
	_env.flag = 0;
	_env.idx  = 0;
	usr_adc_average_value_get();
	printf_e("%-5d %-6d %-6d %-6d %-6d\r\n", ++_env.cnt, _env.avg[0],
	                                 _env.avg[1], _env.avg[2], _env.avg[3]);

	while (1) {
		ald_delay_ms(1000);
		usr_adc_insert_start();
		while (!_env.flag);
		_env.flag = 0;
		_env.idx  = 0;
		usr_adc_average_value_get();
		printf_e("%-5d %-6d %-6d %-6d %-6d\r\n", ++_env.cnt, _env.avg[0],
	                                   _env.avg[1], _env.avg[2], _env.avg[3]);
	}
}

/**
  * @}
  */

/**
  * @}
  */
