/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    25 Jul 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          25 Jul 2022     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"
uint8_t uart1_interrupt = 0;
uint8_t tx_buf[8];
uint8_t rx_buf[8];
uint8_t size = 0;
/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLFREQ_48MHz,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_PLL,
    (MD_RCU_HRCON | MD_RCU_PLLON),
};

md_gpio_inittypedef GPIOB_PIN0_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_0,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF3
};

md_gpio_inittypedef GPIOB_PIN1_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_1,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF3
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

/* Private define -------------------------------------------------------------*/
/* Private macro --------------------------------------------------------------*/
/* Private variable -----------------------------------------------------------*/
/* Private function prototypes ------------------------------------------------*/
void Iomux(void);
void SysPeriInit(void);

/* Private functions ----------------------------------------------------------*/
/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    uint32_t i = 0;
    uint8_t temp = 0;
    bool key_flag = 0;
    bool frame_flag = 0;
    int key_num = 0;

    __disable_irq();

    md_rcu_pll_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    SysPeriInit();
    Iomux();
    bsp_led_init();
    bsp_key_init();
    md_uart_init(UART1, &uart_initStruct);
    NVIC_EnableIRQ(UART1_IRQn);
    size = 1;
    md_uart_set_rx_threshold(UART1, MD_UART_RX_FIFO_ONE);
    md_uart_enable_it_rfth(UART1);
    __enable_irq();

    printf("SystemFrequency_SysClk:%d\r\n", SystemFrequency_SysClk);
    printf("SystemFrequency_AHBClk:%d\r\n", SystemFrequency_AHBClk);
    printf("SystemFrequency_APBClk:%d\r\n", SystemFrequency_APBClk);
    printf("\r\n");

    printf("Press UP key increase RX threshold\r\n");
    printf("Press DOWN key decrease RX threshold\r\n");

    printf("UART MD Rece_Send_By_IT Example(Rx threshold 1)\r\n");

    while (1)
    {
        if (bsp_key_up_gpio_get() && key_flag == 0)
        {
            key_flag = 1;
            frame_flag = 1;

            if (key_num < 3)
                key_num++;
        }
        else if (bsp_key_down_gpio_get() && key_flag == 0)
        {
            key_flag = 1;
            frame_flag = 1;

            if (key_num > 0)
                key_num--;
        }
        else if (bsp_key_up_gpio_get() == 0 && bsp_key_down_gpio_get() == 0)
        {
            key_flag = 0;
        }

        if (frame_flag == 1)
        {
            frame_flag = 0;
            md_uart_disable_it_rfth(UART1);

            switch (key_num)
            {
                case 0:
                    size = 1;
                    break;

                case 1:
                    size = 2;
                    break;

                case 2:
                    size = 4;
                    break;

                case 3:
                    size = 8;
                    break;
            }

            if (size == 1)
                md_uart_set_rx_threshold(UART1, MD_UART_RX_FIFO_ONE);
            else if (size == 2)
                md_uart_set_rx_threshold(UART1, MD_UART_RX_FIFO_ONE_QUARTER);
            else if (size == 4)
                md_uart_set_rx_threshold(UART1, MD_UART_RX_FIFO_ONE_HALF);
            else if (size == 6)
                md_uart_set_rx_threshold(UART1, MD_UART_RX_FIFO_TWO_FULL);

            md_uart_enable_it_rfth(UART1);

            printf("\r\nUART MD Rece_Send_By_IT Example(Rx threshold %d)\r\n", size);
        }

        if (uart1_interrupt == 1)
        {
            for (i = 0 ; i < size; i++)
            {
                md_uart_send(UART1, rx_buf[i]);
            }

            uart1_interrupt = 0;

            temp ^= 1;

            if (temp)
            {
                bsp_Led1_On();
                bsp_Led2_Off();
            }
            else
            {
                bsp_Led1_Off();
                bsp_Led2_On();
            }
        }
    }
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);

    md_rcu_enable_uart1(RCU);
}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB0: UART1_TX.
  *         PB1: UART1_RX (Internal weak pull-up).
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOB, &GPIOB_PIN0_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN1_Init);
}


/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY)));  /* Tx FIFO empty */

    UART1->TXDATA = ch;            /* Sent byte */
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


