/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    24 Aug 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          24 Aug 2022     AE TEAM         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */
/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLFREQ_48MHz,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_PLL,
    (MD_RCU_HRCON | MD_RCU_PLLON),
};

md_gpio_inittypedef GPIOB_PIN0_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_0,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF3
};

md_gpio_inittypedef GPIOB_PIN1_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_1,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF3
};

md_gpio_inittypedef GPIOC_PIN0_Init =    /**< GP32C4T1_CH1 init structure */
{
    MD_GPIO_PIN_0,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF5
};

md_gpio_inittypedef GPIOC_PIN1_Init =    /**< GP32C4T1_CH2 init structure */
{
    MD_GPIO_PIN_1,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF5
};

md_gpio_inittypedef GPIOC_PIN2_Init =    /**< GP32C4T1_CH3 init structure */
{
    MD_GPIO_PIN_2,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF5
};

md_gpio_inittypedef GPIOC_PIN3_Init =    /**< GP32C4T1_CH4 init structure */
{
    MD_GPIO_PIN_3,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF5
};

md_gpio_inittypedef GPIOD_PIN2_Init =    /**< GP16C2T1_CH1 init structure */
{
    MD_GPIO_PIN_2,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF5
};

md_gpio_inittypedef GPIOD_PIN3_Init =    /**< GP16C2T1_CH2 init structure */
{
    MD_GPIO_PIN_3,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF5
};

md_gpio_inittypedef GPIOD_PIN8_Init =    /**< GP16C2T2_CH1 init structure */
{
    MD_GPIO_PIN_8,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF6
};

md_gpio_inittypedef GPIOD_PIN9_Init =    /**< GP16C2T2_CH2 init structure */
{
    MD_GPIO_PIN_9,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF6
};

md_gpio_inittypedef GPIOD_PIN12_Init =    /**< GP16C2T3_CH1 init structure */
{
    MD_GPIO_PIN_12,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF4
};

md_gpio_inittypedef GPIOD_PIN14_Init =    /**< GP16C2T3_CH2 init structure */
{
    MD_GPIO_PIN_14,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF4
};

md_gpio_inittypedef GPIOC_PIN9_Init =    /**< GP16C2T4_CH1 init structure */
{
    MD_GPIO_PIN_9,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF6
};

md_gpio_inittypedef GPIOC_PIN11_Init =    /**< GP16C2T4_CH2 init structure */
{
    MD_GPIO_PIN_11,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF6
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

/* Private define -------------------------------------------------------------*/
/* Private macro --------------------------------------------------------------*/
/* Private variable -----------------------------------------------------------*/
/* Private function prototypes ------------------------------------------------*/
void Iomux(void);
void SysPeriInit(void);
void TimerInit(void);

/* Private functions ----------------------------------------------------------*/
/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    __disable_irq();

    md_rcu_pll_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    SysPeriInit();
    Iomux();
    md_uart_init(UART1, &uart_initStruct);
    md_tick_init(MD_SYSTICK_CLKSRC_HCLK);
    TimerInit();
    __enable_irq();

    printf("PC0: PWM GP32C4T1_CH1.\r\n");
    printf("PC1: PWM GP32C4T1_CH2.\r\n");
    printf("PC2: PWM GP32C4T1_CH3.\r\n");
    printf("PC3: PWM GP32C4T1_CH4.\r\n");
    printf("PD2: PWM GP16C2T1_CH1.\r\n");
    printf("PD3: PWM GP16C2T1_CH2.\r\n");
    printf("PD8: PWM GP16C2T2_CH1.\r\n");
    printf("PD9: PWM GP16C2T2_CH2.\r\n");
    printf("PD12:PWM GP16C2T3_CH1.\r\n");
    printf("PD14:PWM GP16C2T3_CH2.\r\n");
    printf("PC9: PWM GP16C2T4_CH1.\r\n");
    printf("PC11:PWM GP16C2T4_CH2.\r\n");


    printf("12 PWM output\r\n");

    while (1)
    {
    }
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);

    md_rcu_enable_uart1(RCU);
    md_rcu_enable_gp32c4t1(RCU);
    md_rcu_enable_gp16c2t1(RCU);
    md_rcu_enable_gp16c2t2(RCU);
    md_rcu_enable_gp16c2t3(RCU);
    md_rcu_enable_gp16c2t4(RCU);
}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB0: UART1_TX.
  *         PB1: UART1_RX (Internal weak pull-up).
  *         PC0: PWM GP32C4T1_CH1.
  *         PC1: PWM GP32C4T1_CH2.
  *         PC2: PWM GP32C4T1_CH3.
  *         PC3: PWM GP32C4T1_CH4.
  *         PD2: PWM GP16C2T1_CH1.
  *         PD3: PWM GP16C2T1_CH2.
  *         PD8: PWM GP16C2T2_CH1.
  *         PD9: PWM GP16C2T2_CH2.
  *         PD12:PWM GP16C2T3_CH1.
  *         PD14:PWM GP16C2T3_CH2.
  *         PC9: PWM GP16C2T4_CH1.
  *         PC11:PWM GP16C2T4_CH2.
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOB, &GPIOB_PIN0_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN1_Init);

    md_gpio_init(GPIOC, &GPIOC_PIN0_Init);
    md_gpio_init(GPIOC, &GPIOC_PIN1_Init);
    md_gpio_init(GPIOC, &GPIOC_PIN2_Init);
    md_gpio_init(GPIOC, &GPIOC_PIN3_Init);

    md_gpio_init(GPIOD, &GPIOD_PIN2_Init);
    md_gpio_init(GPIOD, &GPIOD_PIN3_Init);

    md_gpio_init(GPIOD, &GPIOD_PIN8_Init);
    md_gpio_init(GPIOD, &GPIOD_PIN9_Init);

    md_gpio_init(GPIOD, &GPIOD_PIN12_Init);
    md_gpio_init(GPIOD, &GPIOD_PIN14_Init);

    md_gpio_init(GPIOC, &GPIOC_PIN9_Init);
    md_gpio_init(GPIOC, &GPIOC_PIN11_Init);
}

/**
  * @brief  Timer Init
  * @note   None
  * @param  None
  * @retval None
  */
void TimerInit(void)
{
    md_timer_set_auto_reload_value_arrv(GP32C4T1, 2399);
    md_timer_set_output_compare1_mode_ch1mod(GP32C4T1, MD_TIMER_OUTPUTMODE_PWMMODE1);
    md_timer_set_output_compare2_mode_ch2mod(GP32C4T1, MD_TIMER_OUTPUTMODE_PWMMODE1);
    md_timer_set_output_compare3_mode_ch3mod(GP32C4T1, MD_TIMER_OUTPUTMODE_PWMMODE1);
    md_timer_set_output_compare4_mode_ch4mod(GP32C4T1, MD_TIMER_OUTPUTMODE_PWMMODE1);
    md_timer_set_capture_compare1_value_ccrv1(GP32C4T1, (md_timer_get_auto_reload_value_arrv(GP32C4T1)) >> 1);
    md_timer_set_capture_compare2_value_ccrv2(GP32C4T1, (md_timer_get_auto_reload_value_arrv(GP32C4T1)) >> 1);
    md_timer_set_capture_compare3_value_ccrv3(GP32C4T1, (md_timer_get_auto_reload_value_arrv(GP32C4T1)) >> 1);
    md_timer_set_capture_compare4_value_ccrv4(GP32C4T1, (md_timer_get_auto_reload_value_arrv(GP32C4T1)) >> 1);
    md_timer_enable_cc1_output_cc1en(GP32C4T1);
    md_timer_enable_cc2_output_cc2en(GP32C4T1);
    md_timer_enable_cc3_output_cc3en(GP32C4T1);
    md_timer_enable_cc4_output_cc4en(GP32C4T1);

    md_timer_set_auto_reload_value_arrv(GP16C2T1, 2399);
    md_timer_set_output_compare1_mode_ch1mod(GP16C2T1, MD_TIMER_OUTPUTMODE_PWMMODE1);
    md_timer_set_output_compare2_mode_ch2mod(GP16C2T1, MD_TIMER_OUTPUTMODE_PWMMODE1);
    md_timer_set_capture_compare1_value_ccrv1(GP16C2T1, (md_timer_get_auto_reload_value_arrv(GP16C2T1)) >> 1);
    md_timer_set_capture_compare2_value_ccrv2(GP16C2T1, (md_timer_get_auto_reload_value_arrv(GP16C2T1)) >> 1);
    md_timer_enable_cc1_output_cc1en(GP16C2T1);
    md_timer_enable_cc2_output_cc2en(GP16C2T1);
    md_timer_enable_main_output_goen(GP16C2T1);
    md_timer_set_slave_trigger_tssel(GP16C2T1, MD_TIMER_TRIGGER_IT4);
    md_timer_set_slave_mode_smods(GP16C2T1, MD_TIMER_SLAVEMODE_TRIGGER);

    md_timer_set_auto_reload_value_arrv(GP16C2T2, 2399);
    md_timer_set_output_compare1_mode_ch1mod(GP16C2T2, MD_TIMER_OUTPUTMODE_PWMMODE1);
    md_timer_set_output_compare2_mode_ch2mod(GP16C2T2, MD_TIMER_OUTPUTMODE_PWMMODE1);
    md_timer_set_capture_compare1_value_ccrv1(GP16C2T2, (md_timer_get_auto_reload_value_arrv(GP16C2T2)) >> 1);
    md_timer_set_capture_compare2_value_ccrv2(GP16C2T2, (md_timer_get_auto_reload_value_arrv(GP16C2T2)) >> 1);
    md_timer_enable_cc1_output_cc1en(GP16C2T2);
    md_timer_enable_cc2_output_cc2en(GP16C2T2);
    md_timer_enable_main_output_goen(GP16C2T2);
    md_timer_set_slave_trigger_tssel(GP16C2T2, MD_TIMER_TRIGGER_IT4);
    md_timer_set_slave_mode_smods(GP16C2T2, MD_TIMER_SLAVEMODE_TRIGGER);

    md_timer_set_auto_reload_value_arrv(GP16C2T3, 2399);
    md_timer_set_output_compare1_mode_ch1mod(GP16C2T3, MD_TIMER_OUTPUTMODE_PWMMODE1);
    md_timer_set_output_compare2_mode_ch2mod(GP16C2T3, MD_TIMER_OUTPUTMODE_PWMMODE1);
    md_timer_set_capture_compare1_value_ccrv1(GP16C2T3, (md_timer_get_auto_reload_value_arrv(GP16C2T3)) >> 1);
    md_timer_set_capture_compare2_value_ccrv2(GP16C2T3, (md_timer_get_auto_reload_value_arrv(GP16C2T3)) >> 1);
    md_timer_enable_cc1_output_cc1en(GP16C2T3);
    md_timer_enable_cc2_output_cc2en(GP16C2T3);
    md_timer_enable_main_output_goen(GP16C2T3);
    md_timer_set_slave_trigger_tssel(GP16C2T3, MD_TIMER_TRIGGER_IT4);
    md_timer_set_slave_mode_smods(GP16C2T3, MD_TIMER_SLAVEMODE_TRIGGER);

    md_timer_set_auto_reload_value_arrv(GP16C2T4, 2399);
    md_timer_set_output_compare1_mode_ch1mod(GP16C2T4, MD_TIMER_OUTPUTMODE_PWMMODE1);
    md_timer_set_output_compare2_mode_ch2mod(GP16C2T4, MD_TIMER_OUTPUTMODE_PWMMODE1);
    md_timer_set_capture_compare1_value_ccrv1(GP16C2T4, (md_timer_get_auto_reload_value_arrv(GP16C2T4)) >> 1);
    md_timer_set_capture_compare2_value_ccrv2(GP16C2T4, (md_timer_get_auto_reload_value_arrv(GP16C2T4)) >> 1);
    md_timer_enable_cc1_output_cc1en(GP16C2T4);
    md_timer_enable_cc2_output_cc2en(GP16C2T4);
    md_timer_enable_main_output_goen(GP16C2T4);
    md_timer_set_slave_trigger_tssel(GP16C2T4, MD_TIMER_TRIGGER_IT4);
    md_timer_set_slave_mode_smods(GP16C2T4, MD_TIMER_SLAVEMODE_TRIGGER);

    md_timer_set_master_trgout_select_mmsel(GP32C4T1, MD_TIMER_MASTERMODE_ENABLE);
    md_timer_enable_master_mode_mscfg(GP32C4T1);
    md_timer_enable_counter_cnten(GP32C4T1);
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY)));  // Tx FIFO empty

    UART1->TXDATA = ch;            // Sent byte
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


