/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    24 Aug 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          24 Aug 2022     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"
#include <math.h>

/** @addtogroup Projects_Examples_MD
  * @{
  */
/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLFREQ_48MHz,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_PLL,
    (RCU_CON_HRCON | RCU_CON_PLLON),
};

md_gpio_inittypedef GPIOB_PIN0_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_0,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF3
};

md_gpio_inittypedef GPIOB_PIN1_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_1,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF3
};

md_gpio_inittypedef GPIOC_PIN0_Init =    /**< GP32C4T1_CH1 init structure */
{
    MD_GPIO_PIN_0,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF5
};

md_gpio_inittypedef GPIOC_PIN1_Init =    /**< GP32C4T1_CH2 init structure */
{
    MD_GPIO_PIN_1,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF5
};

md_gpio_inittypedef GPIOC_PIN2_Init =    /**< GP32C4T1_CH3 init structure */
{
    MD_GPIO_PIN_2,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF5
};

md_gpio_inittypedef GPIOC_PIN3_Init =    /**< GP32C4T1_CH4 init structure */
{
    MD_GPIO_PIN_3,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF5
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

/* Private define -------------------------------------------------------------*/
/* Private macro --------------------------------------------------------------*/
/* Private variable -----------------------------------------------------------*/
/* Private function prototypes ------------------------------------------------*/
void Iomux(void);
void SysPeriInit(void);
void TimerInit(void);

/* Private functions ----------------------------------------------------------*/

/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    uint32_t psc;
    uint32_t freq, duty;

    __disable_irq();

    md_rcu_pll_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    SysPeriInit();
    Iomux();
    bsp_led_init();
    bsp_key_init();
    md_uart_init(UART1, &uart_initStruct);
    NVIC_EnableIRQ((IRQn_Type) GP32C4T1_IRQn);
    TimerInit();
    __enable_irq();

    printf("Timer input mode demo\r\n");
    printf("Input IO PIN : PC0\r\n");
    printf("Input signal: square wave, frequency : 1Hz ~ 1000Hz\r\n");

    psc = md_timer_get_prescaler_value_pscv(GP32C4T1) + 1;

    while (1)
    {

        freq = (int)(double)SystemFrequency_SysClk / (double)(psc * (md_timer_get_capture_compare1_value_ccrv1(GP32C4T1) + 1));
        duty = (int)round((100 * ((double)(md_timer_get_capture_compare2_value_ccrv2(GP32C4T1) + 1) / (double)(md_timer_get_capture_compare1_value_ccrv1(GP32C4T1) + 1))));

        if (md_timer_is_active_it_upd(GP32C4T1))
        {
            md_timer_clear_it_upd(GP32C4T1);
            md_timer_clear_it_ch1(GP32C4T1);
            md_timer_clear_it_ch2(GP32C4T1);
            printf("freq less than 1Hz\r\n");
        }
        else if (md_timer_is_active_it_ch1(GP32C4T1) && md_timer_is_active_it_ch2(GP32C4T1))
        {
            md_timer_clear_it_ch1(GP32C4T1);
            md_timer_clear_it_ch2(GP32C4T1);
            printf("freq = %dHz  duty = %d%%\r\n", freq, duty);
        }

    }
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);

    md_rcu_enable_uart1(RCU);
    md_rcu_enable_gp32c4t1(RCU);
}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB0: UART1_TX.
  *         PB1: UART1_RX (Internal weak pull-up).
  *         PC0: PWM GP32C4T1_CH1.
  *         PC1: PWM GP32C4T1_CH2.
  *         PC2: PWM GP32C4T1_CH3.
  *         PC3: PWM GP32C4T1_CH4.
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOB, &GPIOB_PIN0_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN1_Init);

    md_gpio_init(GPIOC, &GPIOC_PIN0_Init);
    md_gpio_init(GPIOC, &GPIOC_PIN1_Init);
    md_gpio_init(GPIOC, &GPIOC_PIN2_Init);
    md_gpio_init(GPIOC, &GPIOC_PIN3_Init);
}

/**
  * @brief  Timer Init
  * @note   None
  * @param  None
  * @retval None
  */
void TimerInit(void)
{
    md_timer_set_prescaler_value_pscv(GP32C4T1, 1);
    md_timer_set_auto_reload_value_arrv(GP32C4T1, 0x2FFFFFF);
    md_timer_set_cc1_func_cc1ssel(GP32C4T1, MD_TIMER_CHMODE_INPUT_DIRECT);
    md_timer_set_cc2_func_cc2ssel(GP32C4T1, MD_TIMER_CHMODE_INPUT_INDIRECT);

    md_timer_set_cc1_input_edge_cc1pol(GP32C4T1, MD_TIMER_OUTPUTPOLARITY_HIGH);
    md_timer_set_cc2_input_edge_cc2pol(GP32C4T1, MD_TIMER_OUTPUTPOLARITY_LOW);
    md_timer_set_slave_mode_smods(GP32C4T1, MD_TIMER_SLAVEMODE_RESET);
    md_timer_set_slave_trigger_tssel(GP32C4T1, MD_TIMER_TRIGGER_I1F);
    md_timer_set_update_source_uersel(GP32C4T1, MD_TIMER_UPDATESOURCE_COUNTER);
    md_timer_enable_cc1_input_cc1en(GP32C4T1);
    md_timer_enable_cc2_input_cc2en(GP32C4T1);
    md_timer_enable_counter_cnten(GP32C4T1);
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY_MSK)));  /* Tx FIFO full */

    UART1->TXDATA = ch;            /* Sent byte */
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


