/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    24 Aug 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          24 Aug 2022     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */
/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLFREQ_48MHz,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_PLL,
    (RCU_CON_HRCON | RCU_CON_PLLON),
};

md_gpio_inittypedef GPIOB_PIN0_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_0,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF3
};

md_gpio_inittypedef GPIOB_PIN1_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_1,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF3
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

/* Private define -------------------------------------------------------------*/
/* Private macro --------------------------------------------------------------*/
/* Private variable -----------------------------------------------------------*/
extern uint8_t counter;
volatile extern uint8_t interrupt;
/* Private function prototypes ------------------------------------------------*/
void Iomux(void);
void SysPeriInit(void);
void TimerInit(void);
void TimerInit_1(void);
/* Private functions ----------------------------------------------------------*/

/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    uint8_t temp = 0;
    __disable_irq();

    md_rcu_pll_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    SysPeriInit();
    Iomux();
    bsp_led_init();
    bsp_key_init();
    md_uart_init(UART1, &uart_initStruct);
    NVIC_EnableIRQ((IRQn_Type) GP32C4T1_IRQn);
    TimerInit();
    __enable_irq();
    printf("Timer interrupt demo\r\n");
    printf("Only UART output, through the timer to control the LED blinking speed\r\n");
    printf("enter any key continue\r\n");

    while (md_uart_is_active_flag_rfnempty(UART1) == 0);

    while (1)
    {
        while (!interrupt);

        interrupt = 0;
        temp ^= 1;

        if (temp)
        {
            bsp_Led1_On();
            bsp_Led2_Off();
            printf("Led_1_On & Led_2_Off\n\r");
        }
        else
        {
            bsp_Led1_Off();
            bsp_Led2_On();
            printf("Led_1_Off & Led_2_On\r\n");
        }

        if (counter == 10)
        {
            printf("change TimerInit 1\r\n");
            TimerInit_1();
        }
        else if (counter == 100)
        {
            printf("change TimerInit\r\n");
            counter = 0;
            TimerInit();
        }
    }
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);

    md_rcu_enable_uart1(RCU);
    md_rcu_enable_gp32c4t1(RCU);
}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB0: UART1_TX.
  *         PB1: UART1_RX (Internal weak pull-up).
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOB, &GPIOB_PIN0_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN1_Init);
}

/**
  * @brief  Timer Init
  * @note   None
  * @param  None
  * @retval None
  */
void TimerInit(void)
{
    md_timer_set_prescaler_value_pscv(GP32C4T1, 4799);
    md_timer_set_auto_reload_value_arrv(GP32C4T1, 10000);
    md_timer_enable_it_upd(GP32C4T1);
    md_timer_enable_counter_cnten(GP32C4T1);
}

/**
  * @brief  Timer Init
  * @note   None
  * @param  None
  * @retval None
  */
void TimerInit_1(void)
{
    md_timer_set_prescaler_value_pscv(GP32C4T1, 479);
    md_timer_set_auto_reload_value_arrv(GP32C4T1, 10000);
    md_timer_enable_it_upd(GP32C4T1);
    md_timer_enable_counter_cnten(GP32C4T1);
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY)));  // Tx FIFO empty

    UART1->TXDATA = ch;            // Sent byte
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


