/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    23 Nov 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          23 Nov 2021     Ginger          the first version
 *          30 Dec 2021     Ginger          Modify the example for PDS board
 *          25 Mar 2022     AE Team         Modify MD Driver
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"


/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLFREQ_32MHz,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_HRC,
    (RCU_CON_HRCON),
};

md_gpio_inittypedef GPIOB_PIN0_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_0,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF3
};

md_gpio_inittypedef GPIOB_PIN1_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_1,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF3
};

md_gpio_inittypedef GPIOD_PIN2_Init =    /**< SPI1_NSS init structure */
{
    MD_GPIO_PIN_2,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF1
};

md_gpio_inittypedef GPIOD_PIN3_Init =    /**< SPI1_SCK init structure */
{
    MD_GPIO_PIN_3,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF1
};

md_gpio_inittypedef GPIOD_PIN4_Init =    /**< SPI1_MISO init structure */
{
    MD_GPIO_PIN_4,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF1
};

md_gpio_inittypedef GPIOD_PIN5_Init =    /**< SPI1_MOSI init structure */
{
    MD_GPIO_PIN_5,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF1
};

md_gpio_inittypedef GPIOC_PIN8_Init =    /**< RST pin init structure */
{
    MD_GPIO_PIN_8,
    MD_GPIO_MODE_OUTPUT,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOC_PIN9_Init =    /**< DC pin init structure */
{
    MD_GPIO_PIN_9,
    MD_GPIO_MODE_OUTPUT,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

md_spi_inittypedef SPI1_InitStruct =    /**< SPI init structure */
{
    MD_SPI_MODE_MASTER,
    MD_SPI_PHASE_1EDGE,
    MD_SPI_POLARITY_LOW,
    MD_SPI_BAUDRATEPRESCALER_DIV4,
    MD_SPI_MSB_FIRST,
    MD_SPI_FULL_DUPLEX,
    MD_SPI_FRAME_FORMAT_8BIT,
    MD_SPI_NSS_HARD,
};

/* Private define -------------------------------------------------------------*/
/* Private macros -------------------------------------------------------------*/
#define GC9203_XSTART 0
#define GC9203_YSTART 22
#define GC9203_WIDTH  220 //220
#define GC9203_HEIGHT 176 //176

// Color definitions
#define GC9203_BLACK   0x0000
#define GC9203_BLUE    0x001F
#define GC9203_RED     0xF800
#define GC9203_GREEN   0x07E0
#define GC9203_CYAN    0x07FF
#define GC9203_MAGENTA 0xF81F
#define GC9203_YELLOW  0xFFE0
#define GC9203_WHITE   0xFFFF

#define RESET_LOW()  md_gpio_set_pin_low(GPIOC, MD_GPIO_PIN_8)
#define RESET_HIGH() md_gpio_set_pin_high(GPIOC, MD_GPIO_PIN_8)
#define DC_LOW()     md_gpio_set_pin_low(GPIOC, MD_GPIO_PIN_9)
#define DC_HIGH()    md_gpio_set_pin_high(GPIOC, MD_GPIO_PIN_9)

/* Private variables ----------------------------------------------------------*/
static const uint16_t init_cmds[] =
{
    0x00ff, 0x5aa5,
    0x00f6, 0x0112,
    0x00ef, 0x1052,
    0x00a5, 0x0780,
    0x0002, 0x0100,
    0x0003, 0x1030,
    0x00f6, 0x0112,
    0x0011, 0x1000,
    0x00eb, 0x0d1f,
    0x00ec, 0x0528,
    0x0050, 0xf380,
    0x0051, 0x110c,
    0x0052, 0x0909,
    0x0053, 0x3706,
    0x0054, 0x4e1f,
    0x0055, 0x1212,
    0x0056, 0x3736,
    0x0057, 0xfc80,
    0x0058, 0x110c,
    0x0059, 0x0909,
    0x005a, 0x3706,
    0x005b, 0x4e19,
    0x005c, 0x1212,
    0x005d, 0x3736,
    0x0007, 0x1013,
    0x0001, 0x011c,
    0x00fe
};

/* Private function prototypes ------------------------------------------------*/
void Iomux(void);
void SysPeriInit(void);

/* Private functions ----------------------------------------------------------*/

void spi_lcd_gc9203_reset()
{
    RESET_HIGH();
    md_tick_waitms(10, 2);
    RESET_LOW();
    md_tick_waitms(10, 2);
    RESET_HIGH();
    md_tick_waitms(10, 12);
}

void spi_lcd_gc9203_write(SPI_TypeDef *SPIx, uint8_t *pData, uint16_t Size)
{
    while (Size--)
    {
        while (md_spi_is_active_flag_txf(SPIx));

        md_spi_set_data_reg(SPIx, *pData++);
    }

    while (!md_spi_is_active_flag_txe(SPIx));

    while (md_spi_is_active_flag_busy(SPIx));
}

void spi_lcd_gc9203_send_command(uint8_t cmd)
{
    uint8_t data[2];
    data[0] = 0x00;
    data[1] = cmd;
    DC_LOW();
    spi_lcd_gc9203_write(SPI1, data, 2);
}

void spi_lcd_gc9203_send_data(uint8_t *buff, uint32_t buff_size)
{
    DC_HIGH();
    spi_lcd_gc9203_write(SPI1, buff, buff_size);
}

void spi_lcd_gc9203_execute_command_list(const uint16_t *addr)
{
    uint8_t numCommands, cmd;
    uint16_t data;
    numCommands = sizeof(init_cmds) / sizeof(init_cmds[0]) / 2;

    while (numCommands--)
    {
        cmd = *addr++;
        spi_lcd_gc9203_send_command(cmd);
        data = *addr++;
        data = ((data << 8) & 0xff00) | ((data >> 8) & 0xff);
        spi_lcd_gc9203_send_data((uint8_t *)&data, 2);
    }
}

void spi_lcd_gc9203_init()
{
    spi_lcd_gc9203_reset();
    spi_lcd_gc9203_execute_command_list(init_cmds);
}

void spi_lcd_gc9203_set_address_window(uint8_t x0, uint8_t y0, uint8_t x1, uint8_t y1)
{
    uint8_t data[2] = {0x00, 0x00};
    // Horizonta address start
    data[1] = x0 + GC9203_XSTART;
    spi_lcd_gc9203_send_command(0x39);
    spi_lcd_gc9203_send_data(data, sizeof(data));

    // Horizonta address end
    data[1] = x1 + GC9203_XSTART;
    spi_lcd_gc9203_send_command(0x38);
    spi_lcd_gc9203_send_data(data, sizeof(data));

    // Vertical address start
    data[1] = y0 + GC9203_YSTART;
    spi_lcd_gc9203_send_command(0x37);
    spi_lcd_gc9203_send_data(data, sizeof(data));

    // Vertical address end
    data[1] = y1 + GC9203_YSTART;
    spi_lcd_gc9203_send_command(0x36);
    spi_lcd_gc9203_send_data(data, sizeof(data));
}

void spi_lcd_gc9203_fill_rectangle(uint16_t x, uint16_t y, uint16_t w, uint16_t h, uint16_t color)
{
    // clipping
    if ((x >= GC9203_WIDTH) || (y >= GC9203_HEIGHT)) return;

    if ((x + w - 1) >= GC9203_WIDTH) w = GC9203_WIDTH - x;

    if ((y + h - 1) >= GC9203_HEIGHT) h = GC9203_HEIGHT - y;

    spi_lcd_gc9203_set_address_window(x, y, x + w - 1, y + h - 1);

    // write to RAM
    spi_lcd_gc9203_send_command(0x22);

    uint8_t data[] = { color >> 8, color & 0xFF };
    DC_HIGH();

    for (y = h; y > 0; y--)
    {
        for (x = w; x > 0; x--)
        {
            spi_lcd_gc9203_write(SPI1, data, sizeof(data));
        }
    }
}

void spi_lcd_gc9203_fill_screen(uint16_t color)
{
    spi_lcd_gc9203_fill_rectangle(0, 0, GC9203_WIDTH, GC9203_HEIGHT, color);
}

/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    __disable_irq();
    md_rcu_pll_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    SysPeriInit();
    Iomux();
    bsp_led_init();
    bsp_key_init();
    md_uart_init(UART1, &uart_initStruct);
    __enable_irq();

    md_tick_init(MD_SYSTICK_CLKSRC_HCLK);
    md_spi_init(SPI1, &SPI1_InitStruct);

    printf("SystemFrequency = %d\r\n", SystemFrequency_SysClk);

    printf("Please connect lcd ic GC9203\r\n");
    printf("SPI NSS PIN:PD02\r\n");
    printf("SPI SCK PIN:PD03\r\n");
    printf("SPI MISO PIN:PD04\r\n");
    printf("SPI MOSI PIN:PD05\r\n");
    printf("RES:PC8\r\n");
    printf("DC(RS):PC9\r\n");
    printf("enter any key continue\r\n\r\n");

    while (md_uart_is_active_flag_rfnempty(UART1) == 0);

    spi_lcd_gc9203_init();

    printf("running ...\r\n");

    while (1)
    {
        spi_lcd_gc9203_fill_screen(GC9203_BLUE);
        spi_lcd_gc9203_fill_screen(GC9203_WHITE);
        spi_lcd_gc9203_fill_screen(GC9203_RED);
    }
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);
    md_rcu_enable_uart1(RCU);
    md_rcu_enable_spi1(RCU);
}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB0: UART1_TX.
  *         PB1: UART1_RX (Internal weak pull-up).
  *         PD2: SPI1_NSS.
  *         PD3: SPI1_SCK.
  *         PD4: SPI1_MISO.
  *         PD5: SPI1_MOSI.
  *         PC8: RES.
  *         PC9: DC(RS).
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOB, &GPIOB_PIN0_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN1_Init);

    md_gpio_init(GPIOD, &GPIOD_PIN2_Init);
    md_gpio_init(GPIOD, &GPIOD_PIN3_Init);
    md_gpio_init(GPIOD, &GPIOD_PIN4_Init);
    md_gpio_init(GPIOD, &GPIOD_PIN5_Init);

    md_gpio_init(GPIOC, &GPIOC_PIN8_Init);
    md_gpio_init(GPIOC, &GPIOC_PIN9_Init);
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY)));  // Tx FIFO empty

    UART1->TXDATA = ch;            // Sent byte
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


