/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    14 Jul 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          14 Jul 2022     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include <string.h>
#include <stdio.h>
#include "main.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLFREQ_48MHz,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_PLL,
    (RCU_CON_HRCON | RCU_CON_PLLON),
};

md_gpio_inittypedef GPIOB_PIN0_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_0,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF3
};

md_gpio_inittypedef GPIOB_PIN1_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_1,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF3
};

md_gpio_inittypedef GPIOD_PIN2_Init =    /**< SPI1_NSS init structure */
{
    MD_GPIO_PIN_2,
    MD_GPIO_MODE_OUTPUT,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF1
};

md_gpio_inittypedef GPIOD_PIN3_Init =    /**< SPI1_SCK init structure */
{
    MD_GPIO_PIN_3,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF1
};

md_gpio_inittypedef GPIOD_PIN4_Init =    /**< SPI1_MISO init structure */
{
    MD_GPIO_PIN_4,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF1
};

md_gpio_inittypedef GPIOD_PIN5_Init =    /**< SPI1_MOSI init structure */
{
    MD_GPIO_PIN_5,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF1
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

md_spi_inittypedef SPI1_InitStruct =    /**< SPI init structure */
{
    MD_SPI_MODE_MASTER,
    MD_SPI_PHASE_1EDGE,
    MD_SPI_POLARITY_LOW,
    MD_SPI_BAUDRATEPRESCALER_DIV2,
    MD_SPI_MSB_FIRST,
    MD_SPI_FULL_DUPLEX,
    MD_SPI_FRAME_FORMAT_8BIT,
    MD_SPI_NSS_HARD,
};

typedef enum
{
    MD_OK      = 0x0U,
    MD_ERROR   = 0x1U,
    MD_BUSY    = 0x2U,
    MD_TIMEOUT = 0x3U,
} md_status_t;

/* Private define -------------------------------------------------------------*/
/* Private macros -------------------------------------------------------------*/
#define FLASH_CS_SET()  md_gpio_set_pin_high(GPIOD, MD_GPIO_PIN_2);
#define FLASH_CS_CLR()  md_gpio_set_pin_low(GPIOD, MD_GPIO_PIN_2);

#define FLASH_WRITE_ENABLE  0x06
#define FLASH_ERASE         0x20
#define FLASH_PROGRAM       0x02
#define FLASH_READ          0x03
#define FLASH_ID            0x9F
#define FLASH_STATUS        0x05

/* Private variables ----------------------------------------------------------*/
static char s_flash_txbuf[32] = "essemi mcu spi flash example!";     /*Length must be less than one page (256 bytes)*/
static char s_flash_rxbuf[32];

/* Private function prototypes ------------------------------------------------*/
void Iomux(void);
void SysPeriInit(void);
static void delay(int i);
/* Private functions ----------------------------------------------------------*/
/**
  * @brief  flash sector erase function
  * @param  addr: Specific address which sector to be erase.
  * @retval Status.
  */
md_status_t flash_sector_erase(uint32_t addr)
{
    uint16_t cnt = 5000, temp;
    uint8_t cmd_buf[4];
    uint8_t i = 0U;

    cmd_buf[0] = FLASH_ERASE;           /*Flash sector erase command*/
    cmd_buf[1] = (addr >> 16) & 0xff;   /*24 bit Flash address*/
    cmd_buf[2] = (addr >> 8) & 0xff;
    cmd_buf[3] = addr & 0xff;

    FLASH_CS_CLR();     /*Choose lower, the selected Flash*/

    md_spi_set_data_reg(SPI1, FLASH_WRITE_ENABLE);

    while ((md_spi_is_active_flag_txe(SPI1) == 0) && (--cnt));

    cnt = 5000;

    while ((!((md_spi_is_active_flag_rxne(SPI1)))) && (--cnt));

    temp = md_spi_get_data_reg(SPI1);
    UNUSED(temp);

    FLASH_CS_SET();    /*Pick up and release of Flash*/

    delay(100);
    FLASH_CS_CLR();

    for (i = 0; i < sizeof(cmd_buf); i++)     /*Send the sector erase instructions and Flash address three bytes*/
    {
        cnt = 5000;
        md_spi_set_data_reg(SPI1, cmd_buf[i]);

        while ((md_spi_is_active_flag_txe(SPI1) == 0) && (--cnt));

        cnt = 5000;

        while ((!((md_spi_is_active_flag_rxne(SPI1)))) && (--cnt));

        temp = md_spi_get_data_reg(SPI1);
        UNUSED(temp);
    }

    FLASH_CS_SET();

    return MD_OK;
}

/**
  * @brief  transmit an amount of data in blocking mode.
  * @param  addr: Specific address which to be write.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be sent
  * @retval Status, see @ref md_status_t.
  */
md_status_t flash_write(uint32_t addr, char *buf, uint8_t size)
{
    uint16_t cnt = 5000, temp;
    uint8_t cmd_buf[4];
    uint8_t i = 0U;

    if (buf == NULL)
        return MD_ERROR;

    cmd_buf[0] = FLASH_PROGRAM;
    cmd_buf[1] = (addr >> 16) & 0xff;
    cmd_buf[2] = (addr >> 8) & 0xff;
    cmd_buf[3] = addr & 0xff;

    FLASH_CS_CLR();     /*Choose lower, the selected Flash*/

    md_spi_set_data_reg(SPI1, (uint8_t)FLASH_WRITE_ENABLE);

    while ((md_spi_is_active_flag_txe(SPI1) == 0) && (--cnt));

    cnt = 5000;

    while ((!((md_spi_is_active_flag_rxne(SPI1)))) && (--cnt));

    temp = md_spi_get_data_reg(SPI1);
    UNUSED(temp);

    FLASH_CS_SET();    /*Pick up and release of Flash*/

    delay(100);
    FLASH_CS_CLR();

    for (i = 0; i < sizeof(cmd_buf); i++)     /*Send the programming instructions and Flash address three bytes*/
    {
        cnt = 5000;
        md_spi_set_data_reg(SPI1, (uint8_t)cmd_buf[i]);

        while ((md_spi_is_active_flag_txe(SPI1) == 0) && (--cnt));

        cnt = 5000;

        while ((!((md_spi_is_active_flag_rxne(SPI1)))) && (--cnt));

        temp = md_spi_get_data_reg(SPI1);
        UNUSED(temp);

    }

    for (i = 0; i < size; i++)  /*To write the data sent to the Flash*/
    {
        cnt = 5000;
        md_spi_set_data_reg(SPI1, (uint8_t)buf[i]);

        while ((md_spi_is_active_flag_txe(SPI1) == 0) && (--cnt));

        cnt = 5000;

        while ((!((md_spi_is_active_flag_rxne(SPI1)))) && (--cnt));

        temp = md_spi_get_data_reg(SPI1);
        UNUSED(temp);
    }

    FLASH_CS_SET();

    return MD_OK;
}

/**
  * @brief  Receive an amount of data in blocking mode.
  * @param  addr: address of flash where want to read.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be received
  * @retval Status, see @ref md_status_t.
  */
md_status_t flash_read(uint32_t addr, char *buf, uint16_t size)
{
    uint16_t cnt = 5000, temp;
    uint8_t cmd_buf[4];
    uint8_t i = 0U;

    if (buf == NULL)
        return MD_ERROR;

    cmd_buf[0] = FLASH_READ;
    cmd_buf[1] = (addr >> 16) & 0xff;
    cmd_buf[2] = (addr >> 8) & 0xff;
    cmd_buf[3] = addr & 0xff;

    FLASH_CS_CLR();     /*Choose lower, the selected Flash*/

    for (i = 0; i < sizeof(cmd_buf); i++)   /*Send the editor & reader instructions and Flash address three bytes*/
    {
        md_spi_set_data_reg(SPI1, (uint8_t)cmd_buf[i]);

        while ((md_spi_is_active_flag_txe(SPI1) == 0) && (--cnt));

        cnt = 5000;

        while ((!((md_spi_is_active_flag_rxne(SPI1)))) && (--cnt));

        temp = md_spi_get_data_reg(SPI1);
        UNUSED(temp);
    }

    for (i = 0; i < size; i++)  /*Send the programming instructions and Flash address three bytes*/
    {
        cnt = 5000;
        md_spi_set_data_reg(SPI1, 0xFF);

        while ((md_spi_is_active_flag_txe(SPI1) == 0) && (--cnt));

        cnt = 5000;

        while ((!((md_spi_is_active_flag_rxne(SPI1)))) && (--cnt));

        buf[i] = md_spi_get_data_reg(SPI1);
    }

    FLASH_CS_SET();

    return MD_OK;
}

/**
  * @brief  wait until flash unbusy.
  * @retval Status, see @ref md_status_t.
  */
md_status_t flash_wait_unbusy(void)
{
    uint16_t cnt = 5000, temp;
    uint8_t status;

    FLASH_CS_CLR(); /*Choose lower, the selected Flash*/

    md_spi_set_data_reg(SPI1, (uint8_t)FLASH_STATUS);

    while ((md_spi_is_active_flag_txe(SPI1) == 0) && (--cnt));

    cnt = 5000;

    while ((!((md_spi_is_active_flag_rxne(SPI1)))) && (--cnt));

    temp = md_spi_get_data_reg(SPI1);
    UNUSED(temp);

    do
    {
        cnt = 5000;
        md_spi_set_data_reg(SPI1, 0xFF);

        while ((md_spi_is_active_flag_txe(SPI1) == 0) && (--cnt));

        cnt = 5000;

        while ((!((md_spi_is_active_flag_rxne(SPI1)))) && (--cnt));

        status = md_spi_get_data_reg(SPI1);
    }
    while (status & 0x01);

    FLASH_CS_SET();

    return MD_OK;
}

/**
  * @brief  Read flash id in blocking mode.
  * @retval flash id.
  */
uint32_t flash_read_id(void)
{
    uint16_t cnt = 5000, temp;

    uint8_t i;

    uint8_t flash_id[4] = {0U};

    flash_id[0] = FLASH_ID;

    FLASH_CS_CLR(); /*Choose lower, the selected Flash*/

    for (i = 0; i < sizeof(flash_id); i++)
    {
        md_spi_set_data_reg(SPI1, flash_id[i]);

        while ((md_spi_is_active_flag_txe(SPI1) == 0) && (--cnt));

        cnt = 5000;

        while ((!((md_spi_is_active_flag_rxne(SPI1)))) && (--cnt));

        temp = md_spi_get_data_reg(SPI1);
        UNUSED(temp);

    }

    for (i = 0; i < 3; i++)
    {
        cnt = 5000;
        md_spi_set_data_reg(SPI1, 0xFF);

        while ((md_spi_is_active_flag_txe(SPI1) == 0) && (--cnt));

        cnt = 5000;

        while ((!((md_spi_is_active_flag_rxne(SPI1)))) && (--cnt));

        flash_id[i] = md_spi_get_data_reg(SPI1);
    }

    FLASH_CS_SET();

    return ((flash_id[0] << 16) | (flash_id[1] << 8) | (flash_id[2]));  /*Manufacturer ID flash_id [0] and device ID flash_id [1]*/
}

/**
  * @brief  delay some time.
  * @retval None.
  */
static void delay(int i)
{
    while (i--) ;
}

/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    uint32_t id;
    md_status_t status;

    __disable_irq();
    md_rcu_pll_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    SysPeriInit();
    Iomux();
    md_uart_init(UART1, &uart_initStruct);
    __enable_irq();

    md_spi_init(SPI1, &SPI1_InitStruct);

    bsp_led_init();

    printf("SystemFrequency = %d\r\n", SystemFrequency_SysClk);

    printf("Please connect PDS_ES_ES+FLASH board or flash IC MX25L6406EM21-12G\r\n");
    printf("SPI NSS PIN:PD02\r\n");
    printf("SPI SCK PIN:PD03\r\n");
    printf("SPI MISO PIN:PD04\r\n");
    printf("SPI MOSI PIN:PD05\r\n");
    printf("enter any key continue\r\n\r\n");

    while (md_uart_is_active_flag_rfnempty(UART1) == 0);

    id = flash_read_id();
    printf("\r\nManufacturer ID is %02x & Device ID is %02x %02x\r\n", (uint8_t)(id >> 16), (uint8_t)(id >> 8), (uint8_t)id);

    printf("Now erase the sector containing address 0...\r\n");
    status = flash_sector_erase(0);

    if (status == MD_OK)
    {
        flash_wait_unbusy();    /*Waiting for erasing*/
        printf("Erase OK!\r\n");
    }

    printf("The date written to flash is -> %s\r\n", s_flash_txbuf);
    status = flash_write(0, s_flash_txbuf, strlen(s_flash_txbuf) + 1);

    if (status == MD_OK)
    {
        flash_wait_unbusy();    /*Waiting for the written to complete*/
        printf("Write OK!\r\n");
    }

    status = flash_read(0, s_flash_rxbuf, strlen(s_flash_txbuf) + 1);   /*Read write data*/
    printf("The data read from flash is  -> %s\r\n", s_flash_rxbuf);

    if (!memcmp(s_flash_txbuf, s_flash_rxbuf, strlen(s_flash_txbuf) + 1))   /*Data is written and read*/
        printf("Read OK!\r\n");
    else
        printf("Read ERROR!\r\n");

    while (1)
        ;
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);
    md_rcu_enable_uart1(RCU);
    md_rcu_enable_spi1(RCU);
}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB0: UART1_TX.
  *         PB1: UART1_RX (Internal weak pull-up).
  *         PD2: SPI1_NSS.
  *         PD3: SPI1_SCK.
  *         PD4: SPI1_MISO.
  *         PD5: SPI1_MOSI.
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOB, &GPIOB_PIN0_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN1_Init);

    md_gpio_init(GPIOD, &GPIOD_PIN2_Init);
    md_gpio_init(GPIOD, &GPIOD_PIN3_Init);
    md_gpio_init(GPIOD, &GPIOD_PIN4_Init);
    md_gpio_init(GPIOD, &GPIOD_PIN5_Init);
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY)));  // Tx FIFO empty

    UART1->TXDATA = ch;            // Sent byte
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


