/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    20 Oct. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          20 Oct. 2023    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include "main.h"
#include "rtc.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private constants ----------------------------------------------------------*/
/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLFREQ_48MHz,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_PLL,
    (MD_RCU_HRCON | MD_RCU_PLLON),
    RCU_LCON_LRCON
};

md_gpio_inittypedef UART_TX_PIN_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_10,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF3
};

md_gpio_inittypedef UART_RX_PIN_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_15,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF3
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_NONE,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

/* Private define -------------------------------------------------------------*/
/* Private macro --------------------------------------------------------------*/
/* Private variable -----------------------------------------------------------*/
uint8_t g_time_input[19];
uint8_t g_updateflag = 0;

struct tm
{
    uint8_t tm_sec;                   /* Seconds.     [0-60] (1 leap second) */
    uint8_t tm_min;                   /* Minutes.     [0-59] */
    uint8_t tm_hour;                  /* Hours.       [0-23] */
    uint8_t tm_mday;                  /* Day.         [1-31] */
    uint8_t tm_mon;                   /* Month.       [0-11] */
    uint8_t tm_year;                  /* Year - 1900.  */
};

const char dec_tab[10] = { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9' };
unsigned char cnt = 0;
unsigned char sec_prev = 0;

struct tm tm;

/* Private function prototypes ------------------------------------------------*/
void Iomux(void);
void SysPeriInit(void);

/* Private functions -----------------------------------------------------------*/

void uart_send(unsigned char *data, unsigned char len)
{
    unsigned char i;

    for (i = 0; i < len; ++i)
    {
        while (!(UART1->STAT & (UART_STAT_TFEMPTY)));  // Tx FIFO empty
        UART1->TXDATA = data[i];
    }
}

void uart_puts_time(void)
{
    uint8_t tmstr[64];
    uint8_t i;
    uint8_t *c;

    for (i = 0; i < sizeof(tmstr); ++i)
        tmstr[i] = 0;

    c = tmstr;

    if (tm.tm_sec == sec_prev)
        return;

    sec_prev = tm.tm_sec;

    *c++ = dec_tab[cnt / 100 % 10];
    *c++ = dec_tab[cnt / 10 % 10];
    *c++ = dec_tab[cnt % 10];
    *c++ = ' ';
    *c++ = ' ';
    *c++ = '2';
    *c++ = '0';
    *c++ = dec_tab[((tm.tm_year + 1900) / 10) % 10];
    *c++ = dec_tab[(tm.tm_year + 1900) % 10];
    *c++ = '-';
    *c++ = dec_tab[(tm.tm_mon / 10) % 10];
    *c++ = dec_tab[tm.tm_mon % 10];
    *c++ = '-';
    *c++ = dec_tab[(tm.tm_mday / 10) % 10];
    *c++ = dec_tab[tm.tm_mday % 10];
    *c++ = ' ';
    *c++ = dec_tab[(tm.tm_hour / 10) % 10];
    *c++ = dec_tab[tm.tm_hour % 10];
    *c++ = ':';
    *c++ = dec_tab[(tm.tm_min / 10) % 10];
    *c++ = dec_tab[tm.tm_min % 10];
    *c++ = ':';
    *c++ = dec_tab[(tm.tm_sec / 10) % 10];
    *c++ = dec_tab[tm.tm_sec % 10];
    *c++ = '\n';
    *c = 0;

    cnt++;

    uart_send(tmstr, c - tmstr);
}

/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    uint8_t buf[] = {"Please input time.\nFor example: 2023-01-01 00:00:00\n"};

    __disable_irq();
    md_rcu_pll_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    SysPeriInit();
    Iomux();
    md_uart_init(UART1, &uart_initStruct);
    md_uart_enable_it_rfnempty(UART1);
    NVIC_SetPriority((IRQn_Type) UART1_IRQn, 0);
    NVIC_EnableIRQ((IRQn_Type) UART1_IRQn);
    __enable_irq();

    ds1302_init();
    printf("%s", buf);

    while (!g_updateflag); //等待第一次接收到输入的时间

    while (1)
    {
        if (g_updateflag)
        {
            t_time.year = (g_time_input[2] - 0x30) * 10 + (g_time_input[3] - 0x30);
            t_time.month = (g_time_input[5] - 0x30) * 10 + (g_time_input[6] - 0x30);
            t_time.day = (g_time_input[8] - 0x30) * 10 + (g_time_input[9] - 0x30);
            t_time.hour = ((g_time_input[11] - 0x30) * 10 + (g_time_input[12] - 0x30));
            t_time.minute = (g_time_input[14] - 0x30) * 10 + (g_time_input[15] - 0x30);
            t_time.second = (g_time_input[17] - 0x30) * 10 + (g_time_input[18] - 0x30);
            ds1302_write_time();
            g_updateflag = 0;
        }

        ds1302_read_time();

        tm.tm_sec = t_time.second;
        tm.tm_min = t_time.minute;
        tm.tm_hour = t_time.hour;
        tm.tm_mday = t_time.day;
        tm.tm_mon = t_time.month;
        tm.tm_year = t_time.year + 2000 - 1900;

        if(tm.tm_sec % 2 == 0)
            uart_puts_time();
    }
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);

    md_rcu_enable_uart1(RCU);
}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB0: UART1_TX.
  *         PB1: UART1_RX (Internal weak pull-up).
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOD, &UART_TX_PIN_Init);
    md_gpio_init(GPIOD, &UART_RX_PIN_Init);
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY)));  // Tx FIFO empty

    UART1->TXDATA = ch;            // Sent byte
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


