/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    22 Jul 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          22 Jul 2022     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct1 =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLFREQ_32MHz,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_HRC,
    (MD_RCU_HRCON),
};

md_rcu_init_typedef rcu_initStruct2 =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLFREQ_32MHz,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_HOSC,
    (MD_RCU_HOSCON),
};

md_rcu_init_typedef rcu_initStruct3 =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLFREQ_32MHz,
    MD_RCU_PPRE_HCLK_DIV_2,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_PLL,
    (MD_RCU_HRCON | MD_RCU_PLLON),
};

md_rcu_init_typedef rcu_initStruct4 =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HOSC,
    MD_RCU_PLLFREQ_48MHz,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_2,
    MD_RCU_SW_SYSCLK_PLL,
    (MD_RCU_HOSCON | MD_RCU_PLLON),
};

md_gpio_inittypedef GPIOB_PIN0_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_0,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF3
};

md_gpio_inittypedef GPIOB_PIN1_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_1,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF3
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

/* Private define -------------------------------------------------------------*/
/* Private macro --------------------------------------------------------------*/
/* Private variable -----------------------------------------------------------*/
/* Private function prototypes ------------------------------------------------*/
void Iomux(void);
void SysPeriInit(void);

/* Private functions ----------------------------------------------------------*/

/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    md_rcu_sys_init(RCU, &rcu_initStruct1);
    SysPeriInit();
    Iomux();
    bsp_led_init();
    bsp_key_init();
    md_uart_init(UART1, &uart_initStruct);

    printf("SystemFrequency = %d\r\n", SystemFrequency_SysClk);
    printf("SystemFrequency_AHBClk = %d\r\n", SystemFrequency_AHBClk);
    printf("SystemFrequency_APBClk = %d\r\n", SystemFrequency_APBClk);

    while (md_uart_is_active_flag_tsbusy(UART1));

    md_rcu_sys_init(RCU, &rcu_initStruct2);
    md_uart_init(UART1, &uart_initStruct);
    printf("SystemFrequency = %d\r\n", SystemFrequency_SysClk);
    printf("SystemFrequency_AHBClk = %d\r\n", SystemFrequency_AHBClk);
    printf("SystemFrequency_APBClk = %d\r\n", SystemFrequency_APBClk);
    md_rcu_disable_hrc(RCU);

    while (md_uart_is_active_flag_tsbusy(UART1));

    md_rcu_pll_init(RCU, &rcu_initStruct3);
    md_rcu_sys_init(RCU, &rcu_initStruct3);
    md_uart_init(UART1, &uart_initStruct);
    printf("SystemFrequency = %d\r\n", SystemFrequency_SysClk);
    printf("SystemFrequency_AHBClk = %d\r\n", SystemFrequency_AHBClk);
    printf("SystemFrequency_APBClk = %d\r\n", SystemFrequency_APBClk);
    md_rcu_disable_hosc(RCU);

    while (md_uart_is_active_flag_tsbusy(UART1));

    md_rcu_set_system_clock_source(RCU, MD_RCU_SW_SYSCLK_HRC);

    md_rcu_pll_init(RCU, &rcu_initStruct4);
    md_rcu_sys_init(RCU, &rcu_initStruct4);
    md_uart_init(UART1, &uart_initStruct);
    printf("SystemFrequency = %d\r\n", SystemFrequency_SysClk);
    printf("SystemFrequency_AHBClk = %d\r\n", SystemFrequency_AHBClk);
    printf("SystemFrequency_APBClk = %d\r\n", SystemFrequency_APBClk);

    while (1)
    {
    }
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);

    md_rcu_enable_uart1(RCU);
}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB0: UART1_TX.
  *         PB1: UART1_RX (Internal weak pull-up).
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOB, &GPIOB_PIN0_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN1_Init);
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY)));  /* Tx FIFO empty */

    UART1->TXDATA = ch;            /* Sent byte */
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


