/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    15 Nov 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          15 Nov 2022     Ginger          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS


/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"
extern uint32_t    intrWAKEUP;
extern uint32_t    intrBKWAKEUP;
/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLFREQ_32MHz,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_HRC,
    (RCU_CON_HRCON),
};

md_gpio_inittypedef GPIOB_PIN0_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_0,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF3
};

md_gpio_inittypedef GPIOB_PIN1_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_1,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF3
};

md_gpio_inittypedef GPIOB_PIN2_Init =    /**< WKUP2 init structure */
{
    MD_GPIO_PIN_2,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF8
};

md_gpio_inittypedef GPIOB_PIN3_Init =    /**< WKUP3 init structure */
{
    MD_GPIO_PIN_3,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF8
};

md_gpio_inittypedef GPIOB_PIN10_Init =    /**< WKUP4 init structure */
{
    MD_GPIO_PIN_10,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF8
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

/* Private define -------------------------------------------------------------*/
#define LP_LEVEL MD_SYSCFG_LPLS_STOP1_MODE

/* Private macro --------------------------------------------------------------*/
/* Private variable -----------------------------------------------------------*/
/* Private function prototypes ------------------------------------------------*/
void Iomux(void);
void SysPeriInit(void);
void EnterSleep(uint8_t);

/* Private functions ----------------------------------------------------------*/

/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    uint8_t temp = 0;
    md_rcu_pll_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    SysPeriInit();
    Iomux();
    bsp_led_init();
    md_uart_init(UART1, &uart_initStruct);
    md_tick_init(MD_SYSTICK_CLKSRC_HCLK);

    printf("Low Power Example\r\n");
    printf("Enter Stop mode and wake up from UP/Mid/DOWN Button\r\n");
    printf("=====================================================\r\n");

    //wait UART1 send end
    while (md_uart_is_active_flag_tsbusy(UART1));

    NVIC_EnableIRQ((IRQn_Type)WAKEUP_IRQn);     //Enable WAKEUP interrupt
    md_exti_set_rts(EXTI, MD_EXTI_EVENT_WAKEUP);
    md_exti_enable_it_wakeup(EXTI);

    md_syscfg_set_lowpower_mode(SYSCFG, LP_LEVEL);
    md_syscfg_enable_wakeup_edge_rising(SYSCFG, MD_SYSCFG_WKEG_WKUP2);
    md_syscfg_enable_wakeup_edge_rising(SYSCFG, MD_SYSCFG_WKEG_WKUP3);
    md_syscfg_enable_wakeup_edge_rising(SYSCFG, MD_SYSCFG_WKEG_WKUP4);

    md_syscfg_enable_wakeup(SYSCFG, MD_SYSCFG_WKEN_WKUP2);
    md_syscfg_enable_wakeup(SYSCFG, MD_SYSCFG_WKEN_WKUP3);
    md_syscfg_enable_wakeup(SYSCFG, MD_SYSCFG_WKEN_WKUP4);

//    md_fc_enable_flash_stop_mode(FC);   // Flash: open stop mode,The current can be reduced about 20uA

    EnterSleep(1);

    while (intrWAKEUP == 0);

    //resetting clock config
    md_rcu_sys_init(RCU, &rcu_initStruct);

    printf("WakeUp from Low Power - Level(%d)\r\n", md_syscfg_get_lowpower_mode(SYSCFG));
    printf("Wakeup Events was : 0x%x\r\n", intrBKWAKEUP);

    if (intrBKWAKEUP == MD_SYSCFG_FG_WKUP2)
        printf("Wake up from UP Button\r\n");
    else if (intrBKWAKEUP == MD_SYSCFG_FG_WKUP3)
        printf("Wake up from DOWN Button\r\n");
    else if (intrBKWAKEUP == MD_SYSCFG_FG_WKUP4)
        printf("Wake up from MID Button\r\n");
    else if (intrBKWAKEUP == MD_SYSCFG_FG_NRST)
        printf("Wake up from RESET Button\r\n");
    else

        printf("=====================================================\r\n");

    printf("\r\n");

    md_rcu_clear_flag_reset_flag(RCU);                   //Clear reset flag

    while (1)
    {
        temp ^= 1;
        md_tick_waitms(10, 10);

        if (temp)
        {
            bsp_Led1_On();
            bsp_Led2_Off();
            bsp_Led3_Off();
            bsp_Led4_Off();
        }
        else
        {
            bsp_Led1_Off();
            bsp_Led2_On();
            bsp_Led3_Off();
            bsp_Led4_Off();
        }
    }
}

/**
  * @brief  EnterSleep
  * @note   Enable sleep mode
  * @param  Mode
  * @retval None
  */
void EnterSleep(uint8_t  mode)
{
    // Ensure we SLEEP - SLEEPDEEP should be clear

    if (mode)       // SCR[2] = 1 (SLEEPDEEP)
        SCB->SCR |= (1UL << 2);
    else            // SCR[2] = 0 (SLEEP)
        SCB->SCR &= ~(1UL << 2);

    //SCB->SCR |= (1UL << 1);    // SLEEPONEXIT
    __WFI();    //Wait For Interrupt
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);

    md_rcu_enable_uart1(RCU);
}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB0: UART1_TX.
  *         PB1: UART1_RX (Internal weak pull-up).
  *         PB2: WKUP2.
  *         PB3: WKUP3.
  *         PB10: WKUP4.
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOB, &GPIOB_PIN0_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN1_Init);

    md_gpio_init(GPIOB, &GPIOB_PIN2_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN3_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN10_Init);
}


/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY)));  // Tx FIFO empty

    UART1->TXDATA = ch;            // Sent byte
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


