/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    9 Aug 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          9 Aug 2022      Ginger          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"


/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLFREQ_48MHz,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_PLL,
    (MD_RCU_HRCON | MD_RCU_PLLON),
};

md_gpio_inittypedef GPIOB_PIN0_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_0,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF3
};

md_gpio_inittypedef GPIOB_PIN1_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_1,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF3
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

md_lcd_init_typedef lcd_initSturct =
{
    MD_LCD_DUTY_1D4,
    MD_LCD_BIAS_1D3,
    MD_LCD_TYPE_A,
    8,
    MD_LCD_UPDCTRL_EVERY_FRAME
};

md_gpio_inittypedef GPIOB_PIN12_Init =    /**< LCD COM0 init structure */
{
    MD_GPIO_PIN_12,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF8
};

md_gpio_inittypedef GPIOB_PIN13_Init =    /**< LCD COM1 init structure */
{
    MD_GPIO_PIN_13,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF8
};

md_gpio_inittypedef GPIOB_PIN14_Init =    /**< LCD COM2 init structure */
{
    MD_GPIO_PIN_14,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF8
};

md_gpio_inittypedef GPIOB_PIN15_Init =    /**< LCD COM3 init structure */
{
    MD_GPIO_PIN_15,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF8
};

md_gpio_inittypedef GPIOC_PIN0_Init =    /**< LCD SEG0 init structure */
{
    MD_GPIO_PIN_0,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF8
};

md_gpio_inittypedef GPIOC_PIN1_Init =    /**< LCD SEG1 init structure */
{
    MD_GPIO_PIN_1,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF8
};

md_gpio_inittypedef GPIOC_PIN2_Init =    /**< LCD SEG2 init structure */
{
    MD_GPIO_PIN_2,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF8
};

md_gpio_inittypedef GPIOC_PIN3_Init =    /**< LCD SEG3 init structure */
{
    MD_GPIO_PIN_3,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF8
};

md_gpio_inittypedef GPIOC_PIN4_Init =    /**< LCD SEG4 init structure */
{
    MD_GPIO_PIN_4,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF8
};

md_gpio_inittypedef GPIOC_PIN5_Init =    /**< LCD SEG5 init structure */
{
    MD_GPIO_PIN_5,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF8
};

md_gpio_inittypedef GPIOC_PIN6_Init =    /**< LCD SEG6 init structure */
{
    MD_GPIO_PIN_6,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF8
};

md_gpio_inittypedef GPIOC_PIN7_Init =    /**< LCD SEG7 init structure */
{
    MD_GPIO_PIN_7,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF8
};

md_lcd_power_init_typedef LCD_Power_init =
{
    MD_RCU_CFG2_LCDCK_LRC,
    MD_RCU_CFG2_LCD_DIV2,
    MD_RCU_CFG2_VLCDCK_HRC,
    8,
    0
};

/* Private define -------------------------------------------------------------*/
/* Private macro --------------------------------------------------------------*/
static uint16_t u16DispCom0, u16DispCom1, u16DispCom2, u16DispCom3;
static uint16_t u16PointCom0 = 0, u16PointCom1 = 0, u16PointCom2 = 0, u16PointCom3 = 0;
//         3 3 2 2  1 1 0 0
const uint16_t Seven_Segment_TABLE[16] =        //n d  B A G F  C E D p
{
    0xDE,                                       //0 0  1 1 0 1  1 1 1 0
    0x88,                                       //1 1  1 0 0 0  1 0 0 0
    0xE6,                                       //2 2  1 1 1 0  0 1 1 0
    0xEA,                                       //3 3  1 1 1 0  1 0 1 0
    0xB8,                                       //4 4  1 0 1 1  1 0 0 0
    0x7A,                                       //5 5  0 1 1 1  1 0 1 0
    0x7E,                                       //6 6  0 1 1 1  1 1 1 0
    0xC8,                                       //7 7  1 1 0 0  1 0 0 0
    0xFE,                                       //8 8  1 1 1 1  1 1 1 0
    0xFA,                                       //9 9  1 1 1 1  1 0 1 0
    0x00,                                       //A    0 0 0 0  0 0 0 0
    0x01,                                       //B .  0 0 0 0  0 0 0 1
};
/* Private variable -----------------------------------------------------------*/
void SetTempLcdData(uint16_t *u16pData)
{
    uint16_t u16DispData;
    uint16_t u16D0, u16D1, u16D2, u16D3;

    u16D0 = u16pData[3];
    u16D0 = Seven_Segment_TABLE[u16D0];
    u16D1 = u16pData[2];
    u16D1 = Seven_Segment_TABLE[u16D1];
    u16D2 = u16pData[1];
    u16D2 = Seven_Segment_TABLE[u16D2];
    u16D3 = u16pData[0];
    u16D3 = Seven_Segment_TABLE[u16D3];

    u16DispData   = u16DispCom0 & 0xFF00;
    u16DispData  |= ((u16D0 & 0x00C0) | ((u16D1 & 0x00C0) >> 2) | ((u16D2 & 0x00C0) >> 4) | ((u16D3 & 0x00C0) >> 6));
    u16DispCom0   = u16DispData;

    u16DispData   = u16DispCom1 & 0xFF00;
    u16DispData  |= (((u16D0 & 0x0030) << 2) | (u16D1 & 0x0030) | ((u16D2 & 0x0030) >> 2) | ((u16D3 & 0x0030) >> 4));
    u16DispCom1   = u16DispData;

    u16DispData   = u16DispCom2 & 0xFF00;
    u16DispData  |= (((u16D0 & 0x000C) << 4) | ((u16D1 & 0x000C) << 2) | (u16D2 & 0x000C) | ((u16D3 & 0x000C) >> 2));
    u16DispCom2   = u16DispData;

    u16DispData   = u16DispCom3 & 0xFF00;
    u16DispData  |= (((u16D0 & 0x0003) << 6) | ((u16D1 & 0x0003) << 4) | ((u16D2 & 0x0003) << 2) | (u16D3 & 0x0003));
    u16DispCom3   = u16DispData;
}

void SetPointLcdData(uint16_t *u16pData)
{
    uint16_t u16DispData;
    uint16_t u16D0 = 0, u16D1 = 0, u16D2 = 0, u16D3 = 0;

    if (u16pData[3] == 0xb)
    {
        u16D0 = u16pData[3];
        u16D0 = Seven_Segment_TABLE[u16D0];
    }

    if (u16pData[2] == 0xb)
    {
        u16D1 = u16pData[2];
        u16D1 = Seven_Segment_TABLE[u16D1];
    }

    if (u16pData[1] == 0xb)
    {
        u16D2 = u16pData[1];
        u16D2 = Seven_Segment_TABLE[u16D2];
    }

    if (u16pData[0] == 0xb)
    {
        u16D3 = u16pData[0];
        u16D3 = Seven_Segment_TABLE[u16D3];
    }

    u16DispData   = u16PointCom0 & 0xFF00;
    u16DispData  |= ((u16D0 & 0x00C0) | ((u16D1 & 0x00C0) >> 2) | ((u16D2 & 0x00C0) >> 4) | ((u16D3 & 0x00C0) >> 6));
    u16PointCom0   = u16DispData;



    u16DispData   = u16PointCom1 & 0xFF00;
    u16DispData  |= (((u16D0 & 0x0030) << 2) | (u16D1 & 0x0030) | ((u16D2 & 0x0030) >> 2) | ((u16D3 & 0x0030) >> 4));
    u16PointCom1   = u16DispData;


    u16DispData   = u16PointCom2 & 0xFF00;
    u16DispData  |= (((u16D0 & 0x000C) << 4) | ((u16D1 & 0x000C) << 2) | (u16D2 & 0x000C) | ((u16D3 & 0x000C) >> 2));
    u16PointCom2   = u16DispData;



    u16DispData   = u16PointCom3 & 0xFF00;
    u16DispData  |= (((u16D0 & 0x0003) << 6) | ((u16D1 & 0x0003) << 4) | ((u16D2 & 0x0003) << 2) | (u16D3 & 0x0003));
    u16PointCom3   = u16DispData;


}
void DispLcdData(void)
{
    md_lcd_set_ram0(LCD, (u16DispCom0) | u16PointCom0);
    md_lcd_set_ram2(LCD, (u16DispCom1) | u16PointCom1);
    md_lcd_set_ram4(LCD, (u16DispCom2) | u16PointCom2);
    md_lcd_set_ram6(LCD, (u16DispCom3) | u16PointCom3);
}
void md_lcd_display_float_number(float u32DispVal)
{
    uint32_t u32Data;
    uint16_t au16DispNum[4];
    uint16_t au16PointNum[4] = {0};
    int i = 0;
    uint32_t decimal_places = 0;
    int temp_DispVal = (int)u32DispVal;

    for (i = 0; i < 3; i++)
    {
        temp_DispVal /= 10;

        if (temp_DispVal == 0)
        {
            decimal_places = 3 - i;
            break;
        }

    }

    if (decimal_places > 0)
    {
        au16PointNum[4 - decimal_places] = 0xB;

        for (i = 0; i < decimal_places; i++)
        {
            u32DispVal *= 10;
        }
    }



    u32Data  = (uint32_t)u32DispVal;

    while (u32Data >= 10000)
    {
        u32Data /= 10;
    }

    au16DispNum[3]  = u32Data % 10;

    u32Data /= 10;

    au16DispNum[2]  = u32Data % 10;

    u32Data /= 10;

    au16DispNum[1]  = u32Data % 10;

    u32Data /= 10;

    au16DispNum[0]  = u32Data % 10;




    SetTempLcdData(au16DispNum);
    SetPointLcdData(au16PointNum);
    DispLcdData();
}

/*
Note:When the light_colon variable is set to non-zero, the colon in the middle of the LCD will light up, otherwise, it will turn off the colon in the middle of the LCD when it is set to 0
*/
void md_lcd_set_timer(int hour, int min, bool light_colon)
{
    uint16_t au16DispNum[4];
    uint16_t au16PointNum[4] = {0};

    if (light_colon)
    {
        au16PointNum[0] = 0xB;
    }
    else
    {
        au16PointNum[0] = 0x0;
    }

    au16DispNum[3]  = min % 10;

    min /= 10;

    au16DispNum[2]  = min % 10;



    au16DispNum[1]  = hour % 10;

    hour /= 10;

    au16DispNum[0]  = hour % 10;

    SetTempLcdData(au16DispNum);
    SetPointLcdData(au16PointNum);
    DispLcdData();
}

/* Private function prototypes ------------------------------------------------*/
void Iomux(void);
void SysPeriInit(void);

/* Private functions ----------------------------------------------------------*/
/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    int current_hour = 2;
    int current_min = 35;
    float current_num0 = 0.222f;
    float current_num1 = 48.97f;
    float current_num2 = 853.8f;
    float current_num3 = 3999.0f;
    int temp = 0;
    bool blinking_flag = 0;//clock colon blinking flag
    int button_flag = 0;
    void (*bsp_led_on[])(void) =
    {
        bsp_Led1_On,
        bsp_Led2_On,
        bsp_Led4_On,
        bsp_Led3_On
    };
    void (*bsp_led_off[])(void) =
    {
        bsp_Led1_Off,
        bsp_Led2_Off,
        bsp_Led4_Off,
        bsp_Led3_Off
    };
    __disable_irq();
    md_rcu_pll_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    SysPeriInit();
    Iomux();
    bsp_key_init();
    bsp_led_init();
    md_uart_init(UART1, &uart_initStruct);
    md_tick_init(MD_SYSTICK_CLKSRC_HCLK);
    md_lcd_init(LCD, &LCD_Power_init, &lcd_initSturct);
    __enable_irq();

    printf("On %s at %s \r\n", __DATE__, __TIME__);
    printf("SystemFrequency_SysClk:%d\r\n", SystemFrequency_SysClk);
    printf("SystemFrequency_AHBClk:%d\r\n", SystemFrequency_AHBClk);
    printf("SystemFrequency_APBClk:%d\r\n", SystemFrequency_APBClk);
    printf("\r\n");

    printf("Press Button function List\r\n");
    printf("Button UP:\tLCD display %.3f\r\n", current_num0);
    printf("Button DOWN:\tLCD display %.2f\r\n", current_num1);
    printf("Button LEFT:\tLCD display %.1f\r\n", current_num2);
    printf("Button RIGHT:\tLCD display %.0f\r\n", current_num3);
    printf("Button MID:\tLCD display time\r\n");

    printf("\r\n============================\r\n");
    printf("Switch the LED light every 1 seconds, and turn around every 4 seconds\r\n");
    printf("A round is defined as LED1->LED2->LED4->LED3\r\n");


    while (1)
    {
        for (; current_hour < 24; current_hour++)
        {
            for (; current_min < 60; current_min++)
            {
                for (temp = 0; temp < 60000; temp++)
                {
                    md_tick_waitms(1, 1);

                    if (bsp_key_up_gpio_get())
                    {
                        button_flag = 0;
                    }
                    else if (bsp_key_down_gpio_get())
                    {
                        button_flag = 1;
                    }
                    else if (bsp_key_left_gpio_get())
                    {
                        button_flag = 2;
                    }
                    else if (bsp_key_right_gpio_get())
                    {
                        button_flag = 3;
                    }
                    else if (bsp_key_mid_gpio_get())
                    {
                        button_flag = 4;
                    }

                    switch (button_flag)
                    {
                        case 0:
                            md_lcd_display_float_number(current_num0);
                            break;

                        case 1:
                            md_lcd_display_float_number(current_num1);
                            break;

                        case 2:
                            md_lcd_display_float_number(current_num2);
                            break;

                        case 3:
                            md_lcd_display_float_number(current_num3);
                            break;

                        case 4:
                            if (temp % 300 == 0)
                            {
                                blinking_flag ^= 1;
                            }

                            md_lcd_set_timer(current_hour, current_min, blinking_flag);
                            break;
                    }

                    //Switch the LED light every 1 seconds, and turn around every 4 seconds
                    //A round is defined as LED1->LED2->LED4->LED3
                    bsp_led_on[((temp / 1000)) % 4]();
                    bsp_led_off[((temp / 1000) + 1) % 4]();
                    bsp_led_off[((temp / 1000) + 2) % 4]();
                    bsp_led_off[((temp / 1000) + 3) % 4]();
                }
            }
        }

        current_hour = 0;
        current_min = 0;
    }
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);

    md_rcu_enable_gp32c4t1(RCU);

    md_rcu_enable_uart1(RCU);
}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB0: UART1_TX.
  *         PB1: UART1_RX (Internal weak pull-up).
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOB, &GPIOB_PIN0_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN1_Init);

    md_gpio_init(GPIOB, &GPIOB_PIN12_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN13_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN14_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN15_Init);

    md_gpio_init(GPIOC, &GPIOC_PIN0_Init);
    md_gpio_init(GPIOC, &GPIOC_PIN1_Init);
    md_gpio_init(GPIOC, &GPIOC_PIN2_Init);
    md_gpio_init(GPIOC, &GPIOC_PIN3_Init);
    md_gpio_init(GPIOC, &GPIOC_PIN4_Init);
    md_gpio_init(GPIOC, &GPIOC_PIN5_Init);
    md_gpio_init(GPIOC, &GPIOC_PIN6_Init);
    md_gpio_init(GPIOC, &GPIOC_PIN7_Init);

}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY)));  // Tx FIFO empty

    UART1->TXDATA = ch;            // Sent byte
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


