/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    10 Aug 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          10 Aug 2022      Ginger          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/


/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"


/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLFREQ_48MHz,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_PLL,
    (MD_RCU_HRCON | MD_RCU_PLLON),
};

md_gpio_inittypedef GPIOB_PIN0_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_0,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF3
};

md_gpio_inittypedef GPIOB_PIN1_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_1,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF3
};

md_gpio_inittypedef GPIOB_PIN3_Init =    /**< CMP_OUT init structure */
{
    MD_GPIO_PIN_3,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF7
};

md_gpio_inittypedef GPIOA_PIN0_Init =    /**< CMP_INP init structure */
{
    MD_GPIO_PIN_0,
    MD_GPIO_MODE_ANALOG,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOA_PIN1_Init =    /**< CMP_INN init structure */
{
    MD_GPIO_PIN_1,
    MD_GPIO_MODE_ANALOG,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

/* Private define -------------------------------------------------------------*/
/* Private macro --------------------------------------------------------------*/
/* Private variable -----------------------------------------------------------*/
/* Private function prototypes ------------------------------------------------*/
void Iomux(void);
void SysPeriInit(void);

/* Private functions ----------------------------------------------------------*/

void cmp_init()
{
    md_cmp_config_typedef config =
    {
        MD_CMP_CFG1_BLANKSEL_CLOSE,
        MD_CMP_CFG1_OUTSEL_NO,
        MD_CMP_CFG1_INNSEL_EXTERNAL,
        MD_CMP_CFG1_INPSEL_EXTERNAL
    };
    md_rcu_enable_cmp(RCU);

    printf("\r\n");

    md_cmp_enable(CMP);
    md_cmp_config(CMP, &config);
    md_cmp_disable_filter(CMP);
    md_cmp_disable_hysteresis(CMP);
    md_cmp_disable_polarity(CMP);
}

/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    __disable_irq();
    md_rcu_pll_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    SysPeriInit();
    Iomux();
    bsp_led_init();
    bsp_key_init();
    md_uart_init(UART1, &uart_initStruct);
    md_tick_init(MD_SYSTICK_CLKSRC_HCLK);
    __enable_irq();

    printf("SystemFrequency_SysClk = %d\r\n", SystemFrequency_SysClk);

    cmp_init();

    while (1)
    {
        md_tick_waitms(10, 100);

        printf("cmp output\r\n");

        if (md_cmp_get_output_level(CMP) == MD_CMP_CFG1_VALUE_OUTPUT_HIGH_LEVEL)
        {
            printf("PA00:CMP+ PA01:CMP- Result = High\r\n");
            bsp_Led1_On();
            bsp_Led2_Off();
        }
        else
        {
            printf("PA00:CMP+ PA01:CMP- Result = Low\r\n");
            bsp_Led1_Off();
            bsp_Led2_On();
        }

        printf("================================================\r\n");
    }
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);

    md_rcu_enable_cmp(RCU);
    md_rcu_enable_uart1(RCU);
}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB0: UART1_TX.
  *         PB1: UART1_RX (Internal weak pull-up).
  *         PA0: CMP_INP.
  *         PA1: CMP_INN.
  *         PB3: CMP_OUT.
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOB, &GPIOB_PIN0_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN1_Init);

    md_gpio_init(GPIOA, &GPIOA_PIN0_Init);
    md_gpio_init(GPIOA, &GPIOA_PIN1_Init);
    md_gpio_init(GPIOA, &GPIOB_PIN3_Init);
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY)));  // Tx FIFO empty

    UART1->TXDATA = ch;            // Sent byte
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


