/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    9 Aug 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          9 Aug 2022      Ginger          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"
#include <math.h>

/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLFREQ_32MHz,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_HRC,
    (RCU_CON_HRCON | RCU_CON_HOSCON),
};

md_gpio_inittypedef GPIOB_PIN0_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_0,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF3
};

md_gpio_inittypedef GPIOB_PIN1_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_1,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF3
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

md_adc_init_typedef ADC_Init =
{
    MD_ADC_CURSEL_LOW,
    MD_ADC_VRRS_1X,
    MD_ADC_INGN_1X,
    MD_ADC_CHPS_CHOP_ENABLE_CDS_ENABLE,
    MD_ADC_CKS_1MHZ,
    MD_ADC_OSR_32768,
    MD_ADC_CIC_ORDER_4
};

md_adc_config_typedef ADC_Cofig =
{
    MD_ADC_INPS_TSP,
    MD_ADC_INNS_TSN,
    MD_ADC_VRPS_VDDA,
    MD_ADC_VRNS_VSSH
};

md_adc_config_typedef ADC1_Cofig =
{
    MD_ADC_INPS_TSN,
    MD_ADC_INNS_TSP,
    MD_ADC_VRPS_VDDA,
    MD_ADC_VRNS_VSSH
};

md_adc_power_init_typedef ADC_Power_init =
{
    ANPWR_VLDO_MODE_ADJUSTABLE_VOLTAGE,
    ANPWR_VLDO_SEL_3P0V,
    ADC_CLOCKSOURCE_HRC,
    2,
    CHOP_CLOCKSOURCE_HRC,
    2
};


/* Private define -------------------------------------------------------------*/
/* Private macro --------------------------------------------------------------*/
/* Private variable -----------------------------------------------------------*/

/* Private function prototypes ------------------------------------------------*/
void Iomux(void);
void SysPeriInit(void);

/* Private functions ----------------------------------------------------------*/
/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    int tsp_tsn=0;
    int tsn_tsp=0;
    double adc_value=0;
    __disable_irq();
    md_rcu_pll_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    SysPeriInit();
    Iomux();
    md_uart_init(UART1, &uart_initStruct);
    md_tick_init(MD_SYSTICK_CLKSRC_HCLK);
    __enable_irq();

    NVIC_EnableIRQ((IRQn_Type)ADC_IRQn); //ADC_IRQHandler

    printf("On %s at %s \r\n", __DATE__, __TIME__);
    printf("SystemFrequency_SysClk:%d\r\n", SystemFrequency_SysClk);
    printf("SystemFrequency_AHBClk:%d\r\n", SystemFrequency_AHBClk);
    printf("SystemFrequency_APBClk:%d\r\n", SystemFrequency_APBClk);
    printf("\r\n");


    //using tempture sensor must be open AFEVBG&AFEVBG chopper
    md_anpwr_enable_vbg(ANPWR);
    md_anpwr_set_vbg_chpcks(ANPWR,MD_ANPWR_VBG_CHPCKS_2MHZ);
    md_anpwr_enable_vbg_chp(ANPWR);

    md_adc_init(ADC, &ADC_Power_init, &ADC_Init, &ADC_Cofig);
   
    while (1)
    {
      while (md_adc_is_active_flag_adc_done(ADC) == 0);
      tsp_tsn = (int32_t)md_adc_get_value(ADC);
      
      md_adc_disable_cic_filter(ADC);
      md_adc_basic_config(ADC,&ADC1_Cofig);
      md_adc_clear_flag_adc_reset(ADC);
      md_adc_clear_flag_cic_filter_reset(ADC);
      md_adc_clear_it_adc_done(ADC);
      md_adc_enable_cic_filter(ADC);

      while (md_adc_is_active_flag_adc_done(ADC) == 0);
      tsn_tsp = (int32_t)md_adc_get_value(ADC);
      
      md_adc_disable_cic_filter(ADC);
      md_adc_basic_config(ADC,&ADC_Cofig);
      md_adc_clear_flag_adc_reset(ADC);
      md_adc_clear_flag_cic_filter_reset(ADC);
      md_adc_clear_it_adc_done(ADC);
      md_adc_enable_cic_filter(ADC);

      adc_value=(double)(tsp_tsn-tsn_tsp)/(double)2;
  
      
      printf("current TS temp:%lf\r\n",(((double)adc_value/(double)md_fc_read_adc_temperature_slope())-(double)290.5));

      
    }
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);
    md_rcu_enable_adc(RCU);
    md_rcu_enable_gp32c4t1(RCU);
    md_rcu_enable_anpwr(RCU);
    md_rcu_enable_uart1(RCU);

}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB0: UART1_TX.
  *         PB1: UART1_RX (Internal weak pull-up).
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOB, &GPIOB_PIN0_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN1_Init);
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY)));  // Tx FIFO empty

    UART1->TXDATA = ch;            // Sent byte
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


