/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    9 Aug 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          9 Aug 2022      Ginger          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"
#include <math.h>

/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLFREQ_32MHz,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_HRC,
    (RCU_CON_HRCON | RCU_CON_HOSCON),
};

md_gpio_inittypedef GPIOB_PIN0_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_0,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF3
};

md_gpio_inittypedef GPIOB_PIN1_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_1,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF3
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

md_adc_init_typedef ADC_Init =
{
    MD_ADC_CURSEL_LOW,
    MD_ADC_VRRS_1X,
    MD_ADC_INGN_1X,
    MD_ADC_CHPS_CHOP_ENABLE_CDS_ENABLE,
    MD_ADC_CKS_1MHZ,
    MD_ADC_OSR_32768,
    MD_ADC_CIC_ORDER_4
};

md_adc_config_typedef ADC_Cofig =
{
    MD_ADC_INPS_AIN2,
    MD_ADC_INNS_AIN3,
    MD_ADC_VRPS_VDDA,
    MD_ADC_VRNS_VSSH
};

//uint32_t   intrADC;
//uint32_t   intrADC_t;
/* Private define -------------------------------------------------------------*/
/* Private macro --------------------------------------------------------------*/
#define PI (acos(-1))
/* Private variable -----------------------------------------------------------*/

/* Private function prototypes ------------------------------------------------*/
void Iomux(void);
void SysPeriInit(void);

/* Private functions ----------------------------------------------------------*/
/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    uint32_t data_freq = 2000;
    uint32_t sampling_frequency = 16000;
    uint32_t cut_off_frequency = 300;
    uint32_t data_real = 0;
    int index = 0;
    __disable_irq();
    md_rcu_pll_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    SysPeriInit();
    Iomux();
    md_uart_init(UART1, &uart_initStruct);
    md_tick_init(MD_SYSTICK_CLKSRC_HCLK);
    __enable_irq();

    NVIC_EnableIRQ(ADC_IRQn);
    md_adc_enable_it_biquad_done(ADC);


    printf("On %s at %s \r\n", __DATE__, __TIME__);
    printf("SystemFrequency_SysClk:%d\r\n", SystemFrequency_SysClk);
    printf("SystemFrequency_AHBClk:%d\r\n", SystemFrequency_AHBClk);
    printf("SystemFrequency_APBClk:%d\r\n", SystemFrequency_APBClk);
    printf("\r\n");

    md_biquad_coefficient_determines(ADC, sampling_frequency, cut_off_frequency, MD_ADC_BIQUAD_2ST_ORDER_LOW_PASS_FILTER);


    while (1)
    {
        for (index = 0; index < sampling_frequency / data_freq; index++)
        {
            md_biquad_coefficient_calculate(&data_real, sin(2 * PI * ((double)data_freq / (double)sampling_frequency)*index) + 0.5);
            md_adc_set_biquad_in_value(ADC, data_real);
            md_adc_clear_flag_biquad_swtrig(ADC);
        }
    }
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);
    md_rcu_enable_adc(RCU);
    md_rcu_enable_gp32c4t1(RCU);

    md_rcu_enable_uart1(RCU);
    md_rcu_set_adc_ckdiv(RCU, 1);
    md_rcu_set_adc_ckcfg(RCU, MD_RCU_CFG2_ADCCK_HRC);
}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB0: UART1_TX.
  *         PB1: UART1_RX (Internal weak pull-up).
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOB, &GPIOB_PIN0_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN1_Init);
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY)));  // Tx FIFO empty

    UART1->TXDATA = ch;            // Sent byte
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


