/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    23 Nov 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          23 Nov 2021     Ginger          the first version
 *          23 Dec 2021     Ginger          Modify the example for PDS board
 *          21 Mar 2022     AE Team         Modify MD Driver
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_HRC,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLFREQ_48MHz,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_HRC,
    (MD_RCU_HRCON),
};

md_rcu_init_typedef rcu_initStruct1 =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_HRC,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLFREQ_48MHz,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_HRC,
    (MD_RCU_HRCON),
    MD_RCU_LOSCON
};

md_gpio_inittypedef GPIOB_PIN0_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_0,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF3
};

md_gpio_inittypedef GPIOB_PIN1_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_1,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF3
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

md_gpio_inittypedef GPIOA_PIN0_Init =    /**< MCO init structure */
{
    MD_GPIO_PIN_0,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};



/* Private define -------------------------------------------------------------*/
#define HRC_COUNT (__HRC/(__LOSC/(1<<md_rcu_get_mco_div(RCU))))
#define HRC_TRIM_VALUE 0x80
/* Private macro --------------------------------------------------------------*/
/* Private variable -----------------------------------------------------------*/
/* Private function prototypes ------------------------------------------------*/
void Iomux(void);
void SysPeriInit(void);
void bsp_adc_init(void);
void reset_tick(void);
uint32_t get_tick(void);
void timer_init(void);
void mco_init(void);
/* Private functions ----------------------------------------------------------*/
/**
  * @brief  IrqInit
  * @param  None
  * @retval None
  */
void IrqInit(void)
{
    NVIC->ICER[0] = 0xFFFFFFFFUL;
}
/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    uint32_t min_abs = 0XFFFFFFFF;
    int end_count = 0;
    int i = 0;
    int best_HRCTRIM_L = 0;
    uint32_t backup_min_abs = min_abs;
    __disable_irq();
    md_rcu_pll_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    SysPeriInit();
    IrqInit();
    Iomux();
    md_uart_init(UART1, &uart_initStruct);

    printf("SYSTEM ON HRC ON\r\n");

    md_rcu_sys_init(RCU, &rcu_initStruct1);
    md_uart_init(UART1, &uart_initStruct);

    printf("LOSC ON\r\n");

    printf("On Optionbyte HRCTRIM_L value:%d\r\n", md_rcu_get_hrc16_32trim(RCU));
    printf("press any key start...\r\n");
    md_uart_receive(UART1);
    printf("test start\r\n");  

    md_rcu_set_hrctrim_source(RCU, MD_RCU_HRCSEL_HRCTRIM);
    md_rcu_set_hrc16_32trim(RCU, HRC_TRIM_VALUE);
    md_rcu_set_mco_div(RCU, MD_RCU_MPRE_MCO_DIV1);
    md_rcu_set_mco_source(RCU, MD_RCU_MSW_MCO_HRC);   

    __enable_irq();
    mco_init();
    timer_init();

    best_HRCTRIM_L = HRC_TRIM_VALUE;

    for (i = HRC_TRIM_VALUE ; i > 0; i >>= 1)
    {
        md_timer_enable_counter_cnten(GP32C4T1);
        md_timer_clear_it_ch1(GP32C4T1);

        while (!md_timer_is_active_it_ch1(GP32C4T1));

        md_timer_clear_it_ch1(GP32C4T1);
        end_count = 0;

        while (!md_timer_is_active_it_ch1(GP32C4T1));

        end_count = md_timer_get_capture_compare1_value_ccrv1(GP32C4T1);
        md_timer_clear_it_ch1(GP32C4T1);
        md_timer_disable_counter_cnten(GP32C4T1);

        if (end_count > HRC_COUNT)
        {
            min_abs = (end_count - HRC_COUNT);
        }
        else
        {
            min_abs = (HRC_COUNT - end_count);
        }

        if (min_abs < backup_min_abs)
        {
            backup_min_abs = min_abs;
            best_HRCTRIM_L = md_rcu_get_hrc16_32trim(RCU);
        }

        if (end_count < HRC_COUNT)
        {

            md_rcu_set_hrc16_32trim(RCU, md_rcu_get_hrc16_32trim(RCU) + (i >> 1));
        }
        else
        {
            md_rcu_set_hrc16_32trim(RCU, md_rcu_get_hrc16_32trim(RCU) - (i >> 1));
        }
    }

    md_rcu_set_hrc16_32trim(RCU, best_HRCTRIM_L);
    md_uart_init(UART1, &uart_initStruct);
    printf("check MCO,MCO push out HRC...\r\n");
    md_rcu_set_mco_div(RCU, MD_RCU_MPRE_MCO_DIV1);
    md_rcu_set_mco_source(RCU, MD_RCU_MSW_MCO_HRC);

//    printf("best HRCTRIM_L value:0x%x\r\n", best_HRCTRIM_L);
    printf("best HRCTRIM_L value:%d\r\n", best_HRCTRIM_L);

    while (1);

}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);

    md_rcu_enable_gp32c4t1(RCU);
    md_rcu_enable_uart1(RCU);
}

void mco_init()
{
    //MCO set source LOSC、div 128 => Fmco=32768Hz/128=256Hz
    md_rcu_set_mco_div(RCU, MD_RCU_MPRE_MCO_DIV128);
    md_rcu_set_mco_source(RCU, MD_RCU_MSW_MCO_LOSC);
}

void timer_init()
{
    md_timer_set_prescaler_value_pscv(GP32C4T1, 0);
    md_timer_set_counter_value_cntv(GP32C4T1, 0);

    //ch1 capture mco setting
    md_timer_set_counter_direction_dirsel(GP32C4T1, MD_TIMER_UPCOUNTER);
    md_timer_set_input_capture1_select(GP32C4T1, MD_TIMER_CH1RMP_MCO);
    md_timer_set_cc1_func_cc1ssel(GP32C4T1, MD_TIMER_CHMODE_INPUT_DIRECT);
    md_timer_set_cc1_input_edge_cc1pol(GP32C4T1, MD_TIMER_OUTPUTPOLARITY_HIGH);
    md_timer_enable_cc1_input_cc1en(GP32C4T1);

    //when capture i1 rising ,reset counter value.
    md_timer_set_slave_mode_smods(GP32C4T1, MD_TIMER_SLAVEMODE_RESET);
    md_timer_set_slave_trigger_tssel(GP32C4T1, MD_TIMER_TRIGGER_I1F);
}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB0: UART1_TX.
  *         PB1: UART1_RX (Internal weak pull-up).
            PA2: AIO2
  *         PA3: AIO3
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOB, &GPIOB_PIN0_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN1_Init);


    md_gpio_init(GPIOA, &GPIOA_PIN0_Init);

}

void reset_tick()
{
    md_tick_set_cvr_current(TICK, 0xffffff);
}

uint32_t get_tick()
{
    //uint:s
    return (0xffffff - md_tick_get_cvr_current(TICK)) / 16000000;
}

uint32_t get_tick(void);

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY_MSK)));  // Tx FIFO empty

    UART1->TXDATA = ch;            // Sent byte
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


