/**********************************************************************************
 *
 * @file    exec_proc.c
 * @brief   define the exection functions for state machine
 *
 * @date    10 Mar 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          10 Mar 2022     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ----------------------------------------------------------------- */

#include "comm_proc.h"
#include "main.h"
#include "micro_boot.h"
/* Private Macros ------------------------------------------------------------ */


/* Public Variables ---------------------------------------------------------- */


/* Private Constants --------------------------------------------------------- */


/* Private function prototypes ----------------------------------------------- */


/* Private Variables--------------------------------------------------------- */

/**
  *@brief enum definition of state machine procedure
  */
typedef struct comm_s
{
    uint8_t      dummydata;
    uint8_t      *comm_tx_buf;
    uint8_t      *comm_rx_buf;
    uint32_t     comm_tx_cnt;
    uint32_t     comm_rx_cnt;
    uint32_t     comm_tx_size;
    uint32_t     comm_rx_size;
} comm_t;

__IO static comm_t s_comm_handler;


/* Private Function ---------------------------------------------------------- */

/**
  * @brief  this function is designed to judge whether date the state machine wanna send has been read out by upper
            this function will always return 0, due to the fact, this function only be called via tbc interrupt when
            all expected data sent out in interrupt program
  * @param  None
  * @retval 0x00 to indicate all data expected to send has been read out by upper,
  *         0x01 to indicate process of send is in progress
  */
static fsm_result_t fsm_is_send_over(void)
{
    return PASS;
}

/**
  * @brief  this function is designed to judge whether date the state machine wanna receive has been completed
            this function will always return 0, due to the fact, this function only be called when all expected data
            received in interrupt program
  * @param  None
  * @retval 0x00 to indicate all data expected to receive has been completed,
  *         0x01 to indicate process of receive is in progress
  */
static fsm_result_t fsm_is_recv_over(void)
{
    return PASS;
}

/**
  * @brief  define the send function.
  * @param  None
  * @retval None
  */
static void fsm_send_data(uint8_t *tx_buf, uint32_t len)
{
    if (s_comm_handler.comm_tx_buf  != tx_buf)
        s_comm_handler.comm_tx_buf  = tx_buf;

    s_comm_handler.comm_tx_size = len;
    s_comm_handler.comm_tx_cnt = 0;

    /* enable commnication to send only */
    md_uart_enable_it_tbc(UART1);
    md_uart_set_send_data8(UART1, *(s_comm_handler.comm_tx_buf + s_comm_handler.comm_tx_cnt));

    s_comm_handler.comm_tx_cnt++;

    /*   */
    return;
}

/**
  * @brief  define the receive function.
  * @param  None
  * @retval None
  */
static void fsm_recv_data(uint8_t *rx_buf, uint32_t len)
{
    //printf("[%s]len:%d\r\n", __FUNCTION__, len);
    if (s_comm_handler.comm_rx_buf != rx_buf)
        s_comm_handler.comm_rx_buf = rx_buf;

    s_comm_handler.comm_rx_size  = len;
    s_comm_handler.comm_rx_cnt = 0;

    /* enable commnication to receive only */
//    md_uart_enable_it_rfth(UART1);

    /*   */
    return;
}


/* Public Function ---------------------------------------------------------- */

/**
  * @brief  assign function pointer related to communication to all the state machine subfunction.
  * @param  None
  * @retval None
  */
void fsm_comm_func_init(void)
{
    g_isp_data.p_send_func    = &fsm_send_data;
    g_isp_data.p_recv_func    = &fsm_recv_data;
    g_isp_data.p_is_send_over = &fsm_is_send_over;
    g_isp_data.p_is_recv_over = &fsm_is_recv_over;
}

/**
  * @brief  function to handle communication interrupt.
  * @param  None
  * @retval None
  */
void fsm_int_exec(void)
{
    /* byte send complete */
    if (md_uart_is_masked_it_tbc(UART1))
    {
        md_uart_clear_it_tbc(UART1);

        /* if all expected bytes has not been sent out then continue to receive */
        if (s_comm_handler.comm_tx_size != s_comm_handler.comm_tx_cnt)
        {
            while (!md_uart_is_active_flag_tfempty(UART1));

            md_uart_set_send_data8(UART1, *(s_comm_handler.comm_tx_buf + s_comm_handler.comm_tx_cnt));
            s_comm_handler.comm_tx_cnt++;

            /* allign timer to default when MCU receive each byte */
            g_isp_data.u_frame_timer = FRAME_INTERVAL;
        }
        else
        {
            md_uart_disable_it_tbc(UART1);
            fsm_send_over();
        }
    }

    /* byte receive */
    while (md_uart_is_active_flag_rfnempty(UART1))
    {
//        printf("rx test\r\n");
        *(s_comm_handler.comm_rx_buf + s_comm_handler.comm_rx_cnt) = (uint8_t)md_uart_get_recv_data8(UART1);
        s_comm_handler.comm_rx_cnt++;

        /* receive over */
        if (s_comm_handler.comm_rx_size == s_comm_handler.comm_rx_cnt)
        {
            fsm_recv_over();
        }

        md_uart_clear_it_rfnempty(UART1);
    }

    /* allign timer to default when MCU receive each byte */
    g_isp_data.u_frame_timer = FRAME_INTERVAL;

//    if (md_uart_is_masked_it_rfth(UART1))
//    {
//        md_uart_clear_it_rfth(UART1);

//        while (md_uart_is_active_flag_rfnempty(UART1))
//        {
//            *(s_comm_handler.comm_rx_buf + s_comm_handler.comm_rx_cnt) = (uint8_t)md_uart_get_recv_data8(UART1);
//            s_comm_handler.comm_rx_cnt++;

//            /* receive over */
//            if (s_comm_handler.comm_rx_size == s_comm_handler.comm_rx_cnt)
//            {
//                md_uart_enable_it_rfth(UART1);

//                fsm_recv_over();
//            }
//        }
//        /* allign timer to default when MCU receive each byte */
//        g_isp_data.u_frame_timer = FRAME_INTERVAL;
//    }
}

/**
  * @brief  function to handle systick interrupt.
  * @param  None
  * @retval None
  */
void fsm_systick_int_exec(void)
{
//    /* if "BootPIN" has been pulled down */
//    if (!(md_gpio_get_inputpinset(GPIOD, MD_GPIO_PIN_3)))
//    {
//        printf("%d\r\n", g_isp_data.u_boot_timer);

//        if (g_isp_data.u_boot_timer != 0)
//        {
//            g_isp_data.u_boot_timer--;

//            if (g_isp_data.u_boot_timer == 0)
//            {
//                g_isp_data.jump_flag = 1;
//            }
//        }
//    }

    /* the time interval of received adjacent bytes in the same frame is more than expected, reset tx/rx FIFO */
    if (g_isp_data.u_frame_timer != 0)
    {
        g_isp_data.u_frame_timer--;

        if (g_isp_data.u_frame_timer == 0)
        {
//            md_uart_set_tx_fifo_reset(UART1);
//            md_uart_set_rx_fifo_reset(UART1);
            fsm_para_init();
            return;
        }
    }
}

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
