/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    19 Sep 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          19 Sep 2022     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes -------------------------------------------------------------------*/
#include "main.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private types --------------------------------------------------------------*/
/**
  * @brief GPIOA init structure.
  */
md_gpio_inittypedef GPIOD_PIN0_Init =    /**< GPIOD init structure */
{
    MD_GPIO_PIN_0,
    MD_GPIO_MODE_OUTPUT,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DS_8mA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOC_PIN2_Init =    /**< GPIOC init structure */
{
    MD_GPIO_PIN_2,
    MD_GPIO_MODE_OUTPUT,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DS_8mA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOC_PIN5_Init =    /**< GPIOC init structure */
{
    MD_GPIO_PIN_5,
    MD_GPIO_MODE_INPUT,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_DOWN,
    MD_GPIO_DS_8mA,
    MD_GPIO_AF0
};

/**
  * @brief GPIOB init structure.
  */
md_gpio_inittypedef GPIOC_PIN4_Init =    /**< GPIOC init structure */
{
    MD_GPIO_PIN_4,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_PULL_UP,
    MD_GPIO_DS_8mA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOC_PIN3_Init =    /**< GPIOC init structure */
{
    MD_GPIO_PIN_3,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DS_8mA,
    MD_GPIO_AF0
};

/**
  * @brief WWDT init structure.
  */
md_wwdt_inittypedef WWDT_Init =
{
    WWDT_PRESCALER_8,
    0x7F,
    0x7F,
    WWDT_EWI_DISABLE
};

md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV_1,
    MD_RCU_MCO_DISABLE,
    MD_RCU_HRCDIV_DIV_1,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SYSCLK_HRC,
    (RCU_CON_HRCON_MSK),
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

/** @addtogroup Projects_Examples_MD
  * @{
  */

/**
* @brief  main.
* @param  None
* @retval None
*/
int main(void)
{
    md_rcu_init_set(RCU, &rcu_initStruct);

    md_rcu_enable_wwdt(RCU);
    md_rcu_enable_uart1(RCU);
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);

    md_gpio_init(GPIOC, &GPIOC_PIN4_Init);
    md_gpio_init(GPIOC, &GPIOC_PIN3_Init);
//    md_gpio_init(GPIOD, &GPIOD_PIN0_Init); /* twinkle(H <--> L) means wwdt reset */
//    md_gpio_init(GPIOC, &GPIOC_PIN2_Init); /* high means wwdt not reset */
//    md_gpio_init(GPIOC, &GPIOC_PIN5_Init); /* detect high/low to control the wwdt reset */

    /* Default High for GPIOD PIN0 and Low for GPIOC PIN2  */
//    md_gpio_set_pin_high(GPIOD, MD_GPIO_PIN_0);
//    md_gpio_set_pin_low(GPIOC, MD_GPIO_PIN_2);

    md_uart_init(UART1, &uart_initStruct);
    bsp_key_init();
    bsp_led_init();
    md_wwdt_init(WWDT, &WWDT_Init);



    /* Wait half duty to low GPIOD PIN0 */
    while (md_wwdt_get_count_value(WWDT) > 0x60) {}

//    md_gpio_set_pin_low(GPIOD, MD_GPIO_PIN_0);

    while (1)
    {
        /* check if GPIOC PIN5 is high */
        if ((bsp_key_mid_gpio_get() == 0)) /* GPIOC PIN5 == High */
        {
//            md_gpio_set_pin_high(GPIOC, MD_GPIO_PIN_2);
//            md_gpio_set_pin_pull(GPIOC, MD_GPIO_PIN_5, MD_GPIO_PULL_UP);
            md_wwdt_set_count_value(WWDT, WWDT_Init.Counter); /* To prevent a reset, the counter must be reloaded when its value is greater than 0x3F */
            bsp_led3_on();
        }
        else /* GPIOC PIN5 != High */
//            md_gpio_set_pin_pull(GPIOC, MD_GPIO_PIN_5, MD_GPIO_PULL_DOWN);
            bsp_led3_off();
    }
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (UART1->STAT & (UART_STAT_TFFULL_MSK));  /* Tx FIFO full */

    UART1->TXBUF = ch;            /* Sent byte */
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/

