/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    19 Sep 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          19 Sep 2022     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes--------------------------------------------------------*/
#include "main.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Public Functions -----------------------------------------------------------*/
/* Public Init Structure ------------------------------------------------------*/
/* Public Macros --------------------------------------------------------------*/
/* Public Constants -----------------------------------------------------------*/
/* Private Variables ----------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV_1,
    MD_RCU_MCO_DISABLE,
    MD_RCU_HRCDIV_DIV_1,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SYSCLK_HRC,
    (RCU_CON_HRCON_MSK),
};
/* Private Macros -------------------------------------------------------------*/
/* Private Constants-----------------------------------------------------------*/

/* Private Functions ----------------------------------------------------------*/
void SysInit(void);
void IrqInit(void);
void iomux(void);
void UARTInit(void);
void TimerInit(void);
/**
  * @brief main.
  * @param none
  * @retval none
  */
int main(void)
{
    volatile int i = 0;
    uint32_t psc;
    uint32_t freq, duty;

    SysInit();
    IrqInit();
    iomux();
    UARTInit();
    TimerInit();
    bsp_key_init();
    bsp_led_init();
    printf("GP32C4T1 input mode demo\r\n");
    printf("Input IO PIN : PD0\r\n");
    printf("Input signal: square wave, frequency : 50 ~ 1000Hz\r\n");

    psc = md_timer_get_pscv(GP32C4T1) + 1;

    while (1)
    {
        freq = SystemFrequency_SysClk / (psc * md_timer_get_ccrv1(GP32C4T1));
        duty =  md_timer_get_ccrv2(GP32C4T1) * 100 / md_timer_get_ccrv1(GP32C4T1);
        printf("freq = %dHz  duty = %d%%\r\n", freq, duty);

        for (i = 0; i < 8000000; i++);
    }
}

void SysInit(void)
{
    md_rcu_init_set(RCU, &rcu_initStruct);

    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_disable_hdiven(RCU);

    md_rcu_disable_suart1en(RCU);
    md_rcu_disable_iwdten(RCU);
    md_rcu_disable_wwdt(RCU);
    md_rcu_disable_bs16t1(RCU);
    md_rcu_enable_gp32c4t1(RCU);

    md_rcu_disable_gp16c2t2(RCU);
    md_rcu_disable_gp16c2t1(RCU);
    md_rcu_enable_uart1(RCU);
    md_rcu_disable_spi1(RCU);
    md_rcu_disable_mmcen(RCU);
    md_rcu_disable_lcden(RCU);
}

void IrqInit(void)
{
    NVIC->ICER[0] = 0xFFFFFFFFUL;
}

void iomux(void)
{
    md_gpio_set_pin_pull(GPIOC, MD_GPIO_PIN_3,   MD_GPIO_PULL_UP);
    md_gpio_set_pin_function0_7(GPIOC, MD_GPIO_PIN_3,   MD_GPIO_AF0);
    md_gpio_set_pin_function0_7(GPIOC, MD_GPIO_PIN_4,   MD_GPIO_AF0);
    md_gpio_set_pin_mode(GPIOC, MD_GPIO_PIN_3,   MD_GPIO_MODE_FUNCTION);
    md_gpio_set_pin_mode(GPIOC, MD_GPIO_PIN_4,   MD_GPIO_MODE_FUNCTION);

    md_gpio_set_pin_function0_7(GPIOD, MD_GPIO_PIN_0, MD_GPIO_AF3);
    md_gpio_set_pin_function0_7(GPIOD, MD_GPIO_PIN_2, MD_GPIO_AF3);
    md_gpio_set_pin_mode(GPIOD, MD_GPIO_PIN_0, MD_GPIO_MODE_FUNCTION);
    md_gpio_set_pin_mode(GPIOD, MD_GPIO_PIN_2, MD_GPIO_MODE_FUNCTION);
}

void UARTInit(void)
{
    md_uart_disable_tx(UART1);
    md_uart_disable_rx(UART1);
    md_uart_set_parity(UART1, MD_UART_LCON_PS_EVEN);
    md_uart_set_stop(UART1, MD_UART_LCON_STOP_1);
    md_uart_set_datawidth(UART1, MD_UART_LCON_DLS_8);
    md_uart_enable_tfrst(UART1);
    md_uart_enable_rfrst(UART1);
    md_uart_set_baudrate_calculate(UART1, 115200);
    md_uart_enable_tx(UART1);
    md_uart_enable_rx(UART1);
}

void TimerInit(void)
{
    md_timer_set_pscv(GP32C4T1, 159);
    md_timer_set_arv(GP32C4T1, 0xffff);
    md_timer_set_chmr1_input_cc1ssel(GP32C4T1, MD_TIMER_CHMODE_INPUT_DIRECT);
    md_timer_set_chmr1_input_cc2ssel(GP32C4T1, MD_TIMER_CHMODE_INPUT_INDIRECT);
    md_timer_set_cc1_input_edge_cc1pol(GP32C4T1, MD_TIMER_OUTPUTPOLARITY_HIGH);
    md_timer_set_cc2_input_edge_cc2pol(GP32C4T1, MD_TIMER_OUTPUTPOLARITY_LOW);
    md_timer_set_slave_mode_smods(GP32C4T1, MD_TIMER_SLAVEMODE_RESET);
    md_timer_set_smcon_tssel(GP32C4T1, MD_TIMER_TRIGGERSELECT_TI1FP1);
    md_timer_enable_cc1_input_cc1en(GP32C4T1);
    md_timer_enable_cc2_input_cc2en(GP32C4T1);
    md_timer_enable_counter_cnten(GP32C4T1);
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t sendchar(uint8_t ch)
{
    while (UART1->STAT & (UART_STAT_TFFULL_MSK));  // Tx FIFO full

    UART1->TXBUF = ch;            // Sent byte
    return (ch);
}

/**
  * @}Projects_Examples_MD
  */
/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
