/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    19 Sep 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          19 Sep 2022     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private constants ----------------------------------------------------------*/
/* Private types --------------------------------------------------------------*/

/* Private variables ----------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV_1,
    MD_RCU_MCO_DISABLE,
    MD_RCU_HRCDIV_DIV_1,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SYSCLK_HRC,
    (RCU_CON_HRCON_MSK),
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};
/* Private macros -------------------------------------------------------------*/
/* Private function prototypes ------------------------------------------------*/

/* Public functions -----------------------------------------------------------*/

/**
  * @brief  IrqInit
  * @param  None
  * @retval None
  */
void IrqInit(void)
{
    NVIC->ICER[0] = 0xFFFFFFFFUL;
}

/**
  * @brief  set irq priority
  * @param  None
  * @retval None
  */
void IrqPriority(void)
{
}

/**
  * @brief  configure i/o multiplexer
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    /*  UART IOMux  */
    md_gpio_set_pin_pull(GPIOC, MD_GPIO_PIN_3,   MD_GPIO_PULL_UP);
    md_gpio_set_pin_function0_7(GPIOC, MD_GPIO_PIN_3,   MD_GPIO_AF0);
    md_gpio_set_pin_function0_7(GPIOC, MD_GPIO_PIN_4,   MD_GPIO_AF0);
    md_gpio_set_pin_mode(GPIOC, MD_GPIO_PIN_3,   MD_GPIO_MODE_FUNCTION);
    md_gpio_set_pin_mode(GPIOC, MD_GPIO_PIN_4,   MD_GPIO_MODE_FUNCTION);

    md_gpio_set_pin_pull(GPIOD, MD_GPIO_PIN_2,   MD_GPIO_PULL_UP);
    md_gpio_set_pin_pull(GPIOD, MD_GPIO_PIN_3,   MD_GPIO_PULL_UP);
    md_gpio_set_pin_function0_7(GPIOD, MD_GPIO_PIN_2,   MD_GPIO_AF0);
    md_gpio_set_pin_function0_7(GPIOD, MD_GPIO_PIN_3,   MD_GPIO_AF0);
    md_gpio_set_pin_mode(GPIOD, MD_GPIO_PIN_2,   MD_GPIO_MODE_OUTPUT);
    md_gpio_set_pin_mode(GPIOD, MD_GPIO_PIN_3,   MD_GPIO_MODE_OUTPUT);
}

/**
  * @brief  System Peri init
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_disable_hdiven(RCU);

    md_rcu_disable_suart1en(RCU);
    md_rcu_disable_iwdten(RCU);
    md_rcu_disable_wwdt(RCU);
    md_rcu_disable_bs16t1(RCU);
    md_rcu_disable_gp32c4t1(RCU);

    md_rcu_disable_gp16c2t2(RCU);
    md_rcu_disable_gp16c2t1(RCU);
    md_rcu_enable_uart1(RCU);
    md_rcu_disable_spi1(RCU);
    md_rcu_disable_mmcen(RCU);
    md_rcu_disable_lcden(RCU);
}

/**
  * @brief  CoreInit.
  * @param  None
  * @retval None
  */
void  CoreInit()
{
    SysPeriInit();
    Iomux();
    md_rcu_init_set(RCU, &rcu_initStruct);
    md_uart_init(UART1, &uart_initStruct);
    md_tick_init(MD_SYSTICK_CLKSRC_MCU);
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (UART1->STAT & (UART_STAT_TFFULL_MSK));  // Tx FIFO full

    UART1->TXBUF = ch;            // Sent byte
    return (ch);
}

/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    uint8_t temp ;
    CoreInit();
    temp = 1;
    printf("SystemFrequency_SysClk:%d\r\n", SystemFrequency_SysClk);
    printf("SystemFrequency_AHBClk:%d\r\n", SystemFrequency_AHBClk);
    printf("SystemFrequency_APBClk:%d\r\n", SystemFrequency_APBClk);
    bsp_key_init();
    bsp_led_init();

    while (1)
    {
        md_tick_waitms(10, 50);
        printf("Tick Waitms\r\n");

        if (temp)
            bsp_led3_off();
        else
            bsp_led3_on();

        temp ^= 0x1;
    }

}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


